# Dynamic CMS & Settings Setup Guide

This guide explains the dynamic CMS, menu, and settings system that has been implemented in the MATIKO ticketing platform.

## Overview

The application now features a fully dynamic system where:
- Business settings (name, logo, contact info) are stored in the database and can be changed via the admin panel
- Admin sidebar and login page automatically reflect the business name from settings
- Frontend layout dynamically displays the business name, logo, and settings
- Menus are managed through the CMS and linked to pages
- Pages are created in the CMS with rich content
- All data is accessible via API endpoints for decoupled frontends

## Features Implemented

### 1. Dynamic Business Settings

**Location:** Settings table in database

**Features:**
- Business name, email, phone, address, description
- Logo and favicon URLs
- SEO settings (title, description, keywords)
- Social media links (Facebook, Twitter, Instagram, LinkedIn)
- Feature flags (guest checkout, payment methods)

**Admin Access:**
- Navigate to Settings resource in Filament admin panel
- Change any setting and it will automatically reflect across the entire application

### 2. Dynamic Admin Panel Branding

**File:** `app/Providers/Filament/AdminPanelProvider.php`

The admin panel now reads the business name from settings and displays it on:
- Admin sidebar
- Login page
- All admin pages

When you change the business name in settings, it immediately updates in the admin panel.

### 3. Dynamic Frontend Layout

**File:** `resources/views/layout.blade.php`

The frontend layout now displays:
- Dynamic business name in header and footer
- Dynamic logo (if set) or generated initial letter
- Dynamic SEO meta tags
- Dynamic business description in footer
- Dynamic favicon

### 4. CMS Pages System

**Resources:**
- Model: `app/Models/Page.php`
- Migration: `database/migrations/2026_01_30_000002_create_pages_table.php`
- Filament Resource: `app/Filament/Resources/Menus/PageResource.php`

**Features:**
- Create pages with rich HTML content
- Auto-generate slugs from titles
- SEO meta fields (title, description, keywords)
- Status (draft/published)
- Templates support
- Soft deletes

**Default Pages Created:**
- How It Works
- Help Center
- Contact Us
- About Us
- Privacy Policy
- Terms of Service
- Refund Policy

### 5. Dynamic Menu System

**Resources:**
- Models: `app/Models/Menu.php`, `app/Models/MenuItem.php`
- Migration: `database/migrations/2026_01_30_000003_create_menus_table.php`
- Filament Resource: `app/Filament/Resources/Menus/MenuResource.php`

**Features:**
- Multiple menu locations (topbar, footer)
- Nested menu items (parent/child hierarchy)
- Link to pages, routes, or custom URLs
- Drag-and-drop ordering
- Active/inactive status

**Menu Types:**
1. **Topbar Menu** - Main navigation in header
2. **Footer Menu** - Footer navigation with nested sections

### 6. API Endpoints for Frontend Integration

**Endpoints:**

```
GET /api/v1/menus
GET /api/v1/menus/{location}
GET /api/v1/pages
GET /api/v1/pages/{slug}
GET /api/v1/settings
GET /api/v1/settings/{group}
```

**Controllers:**
- `app/Http/Controllers/Api/MenuController.php`
- `app/Http/Controllers/Api/PublicPageController.php`
- `app/Http/Controllers/Api/PublicSettingsController.php`

## Setup Instructions

### Step 1: Run Migrations

```bash
php artisan migrate
```

This creates the `settings`, `pages`, `menus`, and `menu_items` tables.

### Step 2: Seed the Database

```bash
php artisan db:seed
```

This will:
1. Create roles and default users
2. Seed default business settings
3. Create default CMS pages
4. Create default menus linked to pages
5. Create sample events and tickets

Or run specific seeders:

```bash
php artisan db:seed --class=SettingsSeeder
php artisan db:seed --class=PagesSeeder
php artisan db:seed --class=DefaultMenusSeeder
```

### Step 3: Access the Admin Panel

1. Visit: `http://your-domain.com/admin`
2. Login with:
   - Email: `superadmin@matiko.com`
   - Password: `password`

### Step 4: Manage Settings

1. In admin panel, navigate to **Settings**
2. Change the business name
3. Add logo and favicon URLs
4. Update contact information
5. Modify SEO settings
6. Save changes

### Step 5: Verify Changes

1. **Admin Panel:** Check that the sidebar and login page show your new business name
2. **Frontend:** Visit the homepage and verify the header, footer, and page title reflect your settings
3. **API:** Test the API endpoints to ensure they return your settings

## Usage Guide

### Managing Business Settings

1. Go to Admin Panel → Settings
2. Filter by group: `business`, `seo`, `social`, `features`
3. Edit any setting value
4. Changes are immediately reflected across the application

### Managing Pages

1. Go to Admin Panel → Pages
2. Create new page or edit existing
3. Set title (slug auto-generates)
4. Add content using rich text editor
5. Set SEO meta fields
6. Publish the page
7. Access via: `http://your-domain.com/page/{slug}`

### Managing Menus

1. Go to Admin Panel → Menus
2. Select a menu to edit
3. Add menu items:
   - **Custom URL:** Enter any URL
   - **Page:** Select a CMS page (auto-resolves to `/page/{slug}`)
   - **Route:** Enter a named route (e.g., `my.tickets`)
4. Create nested items by setting a parent
5. Drag to reorder items
6. Save changes

### API Integration

For a decoupled frontend (React, Vue, Next.js), use these endpoints:

**Get Topbar Menu:**
```javascript
fetch('/api/v1/menus/topbar')
  .then(res => res.json())
  .then(data => console.log(data));
```

**Get Page Content:**
```javascript
fetch('/api/v1/pages/about-us')
  .then(res => res.json())
  .then(data => console.log(data));
```

**Get Public Settings:**
```javascript
fetch('/api/v1/settings')
  .then(res => res.json())
  .then(data => console.log(data));
```

## File Structure

```
app/
├── Models/
│   ├── Setting.php
│   ├── Page.php
│   ├── Menu.php
│   └── MenuItem.php
├── Services/
│   └── SettingsService.php
├── View/Composers/
│   ├── MenuComposer.php
│   └── SettingsComposer.php
├── Http/Controllers/Api/
│   ├── MenuController.php
│   ├── PublicPageController.php
│   └── PublicSettingsController.php
├── Filament/Resources/
│   ├── SettingResource.php
│   └── Menus/
│       ├── MenuResource.php
│       └── PageResource.php
└── Providers/
    ├── AppServiceProvider.php
    └── Filament/AdminPanelProvider.php

database/
├── migrations/
│   ├── 2026_01_30_000001_create_settings_table.php
│   ├── 2026_01_30_000002_create_pages_table.php
│   └── 2026_01_30_000003_create_menus_table.php
└── seeders/
    ├── SettingsSeeder.php
    ├── PagesSeeder.php
    └── DefaultMenusSeeder.php

resources/views/
└── layout.blade.php (updated with dynamic settings)

routes/
└── api.php (new public endpoints added)
```

## Testing Checklist

- [ ] Settings seeded correctly
- [ ] Admin panel shows business name from settings
- [ ] Login page shows business name from settings
- [ ] Frontend header shows business name
- [ ] Frontend footer shows business name
- [ ] Logo displays if set (or initial letter if not)
- [ ] SEO meta tags use settings values
- [ ] Pages are accessible via `/page/{slug}`
- [ ] Menus display correctly in frontend
- [ ] Menu items link to correct pages
- [ ] API endpoints return correct data
- [ ] Changing business name updates everywhere

## Troubleshooting

### Settings not showing in admin panel
- Run: `php artisan db:seed --class=SettingsSeeder`
- Clear cache: `php artisan cache:clear`

### Menu items not linking to pages
- Ensure pages are published
- Verify menu item `url_type` is set to 'page'
- Check that `page_id` is correctly set

### Frontend still shows "MATIKO"
- Clear view cache: `php artisan view:clear`
- Clear application cache: `php artisan cache:clear`
- Restart the server

### API endpoints not working
- Clear route cache: `php artisan route:clear`
- Check that controllers are in correct namespace
- Verify routes are registered in `routes/api.php`

## Advanced Customization

### Adding New Settings

```php
Setting::create([
    'key' => 'business.tagline',
    'value' => 'Your event, simplified',
    'group' => 'business',
    'type' => 'string',
    'is_encrypted' => false,
]);
```

### Creating Custom Page Templates

1. Add template to `resources/views/pages/`
2. Update Page model or resource to allow selection
3. Modify PageController to render appropriate template

### Adding Menu Locations

1. Update menu migration to add new location enum value
2. Create new menu in admin panel with that location
3. Add MenuComposer to inject new menu in views
4. Display in layout or specific views

## Security Notes

- Sensitive settings (payment keys, API secrets) should use `is_encrypted => true`
- Public API endpoints only expose non-sensitive settings
- Settings Service automatically encrypts/decrypts based on flag
- Admin panel is protected by Filament authentication

## Performance Optimization

The SettingsService uses caching (1 hour TTL) to reduce database queries. To manually clear settings cache:

```php
app(SettingsService::class)->flush();
```

Or via Artisan:

```bash
php artisan cache:forget settings.*
```

## Support

For questions or issues, contact the development team or refer to the Laravel and Filament documentation.
