# PesaPal Payment Integration - Implementation Tasks

## Project Overview
Implement PesaPal API v3 as an alternative payment method to M-Pesa/Daraja payments for the MATIKO event ticketing system.

## Test Credentials
**Source**: https://developer.pesapal.com/api3-demo-keys.txt

### Available Test Merchant Accounts:
- **Kenya**:
  - Consumer Key: `qkio1BGGYAXTu2JOfm7XSXNruoZsrqEW`
  - Consumer Secret: `osGQ364R49cXKeOYSpaOnT++rHs=`

- **Uganda**:
  - Consumer Key: `TDpigBOOhs+zAl8cwH2Fl82jJGyD8xev`
  - Consumer Secret: `1KpqkfsMaihIcOlhnBo/gBZ5smw=`

- **Tanzania**:
  - Consumer Key: `ngW+UEcnDhltUc5fxPfrCD987xMh3Lx8`
  - Consumer Secret: `q27RChYs5UkypdcNYKzuUw460Dg=`

- **Rwanda**:
  - Consumer Key: `wCGzX1fNzvtI5lMR5M4AxvxBmLpFgZzp`
  - Consumer Secret: `uU7R9g2IHn9dkrKDVIfcPppktIo=`

- **Zambia**:
  - Consumer Key: `v988cq7bMB6AjktYo/drFpe6k2r/y7z3`
  - Consumer Secret: `3p0F/KcY8WAi36LntpPf/Ss0MhQ=`

- **Malawi**:
  - Consumer Key: `htMsEFfIVHfhqBL9O0ykz8wuedfFyg1s`
  - Consumer Secret: `DcwkVNIiyijNWn1fdL/pa4K6khc=`

- **Zimbabwe**:
  - Consumer Key: `vknEWEEFeygxAX+C9TPOhvkbkPsj8qXK`
  - Consumer Secret: `MOOP31smKijvusQbNXn/s7m8jC8=`

## Resources & Documentation
- **Official PesaPal API v3 Docs**: https://developer.pesapal.com/
- **Test Base URL**: `https://cybqa.pesapal.com/pesapalv3`
- **Production Base URL**: `https://pay.pesapal.com/v3/api`
- **Laravel Package**: [njoguamos/laravel-pesapal](https://github.com/njoguamos/laravel-pesapal)
- **Implementation Guide**: [Medium - PesaPal Gateway with Laravel](https://medium.com/@joshuawilfred/pesapal-payment-gateway-integration-with-rilt-stack-react-inertia-laravel-tailwindcss-8a8de8a71082)

## Current System Analysis
### Existing Files:
- **Migration**: `database/migrations/2026_01_14_160000_create_mpesa_transactions_table.php`
- **Controller**: `app/Http/Controllers/Api/PaymentController.php`
- **Model**: `app/Models/MpesaTransaction.php`
- **Order Model**: `app/Models/Order.php`

### Current Order Structure:
- `order_number`
- `user_id`
- `event_id`
- `amount`
- `quantity`
- `total_amount`
- `payment_status`
- `payment_reference`
- `ticket_category_id`
- `phone_number`

---

## Implementation Tasks

### Phase 1: Setup & Dependencies
- [ ] **1.1** Install PesaPal Laravel package
  - Run: `composer require njoguamos/laravel-pesapal`
  - Alternative: `composer require knox/pesapal` or custom implementation

- [ ] **1.2** Publish package configuration
  - Run: `php artisan vendor:publish --tag=pesapal-config`

- [ ] **1.3** Update environment variables
  - Add to `.env`:
    ```
    PESAPAL_CONSUMER_KEY=qkio1BGGYAXTu2JOfm7XSXNruoZsrqEW
    PESAPAL_CONSUMER_SECRET=osGQ364R49cXKeOYSpaOnT++rHs=
    PESAPAL_BASE_URL=https://cybqa.pesapal.com/pesapalv3
    PESAPAL_IPN_URL=${APP_URL}/api/pesapal/ipn
    PESAPAL_CALLBACK_URL=${APP_URL}/api/pesapal/callback
    ```

- [ ] **1.4** Add Kenya merchant credentials as default (can expand later)

---

### Phase 2: Database Schema
- [ ] **2.1** Create `pesapal_transactions` migration
  - Similar structure to `mpesa_transactions`
  - Required fields:
    - `id` (primary key)
    - `order_tracking_id` (PesaPal unique ID) - indexed
    - `merchant_reference` (your order number) - indexed
    - `order_id` (foreign key to orders table)
    - `amount` (decimal 10,2)
    - `currency` (default: KES)
    - `status` (PENDING, COMPLETED, FAILED, CANCELLED)
    - `payment_method` (MPESA, CARD, BANK, etc.)
    - `description` (text)
    - `callback_data` (json - store full callback)
    - `ipn_data` (json - store IPN notifications)
    - `iframe_url` (text - payment iframe URL)
    - `redirect_url` (text - redirect URL)
    - `confirmation_code` (nullable - payment confirmation)
    - `payment_status_description` (text)
    - `created_at`, `updated_at`
    - `completed_at` (nullable timestamp)

- [ ] **2.2** Create `PesapalTransaction` model
  - Relationship to `Order` model
  - Fillable attributes
  - Casts for JSON fields
  - Status constants/enums

- [ ] **2.3** Run migration
  - `php artisan migrate`

---

### Phase 3: Service Layer
- [ ] **3.1** Create `app/Services/PesapalService.php`
  - Authentication method (get OAuth token)
  - Register IPN URL
  - Submit order request
  - Get transaction status
  - Get registered IPNs

- [ ] **3.2** Implement authentication flow
  - OAuth 2.0 token generation
  - Token caching (store in cache for 5 minutes)
  - Automatic token refresh

- [ ] **3.3** Implement order submission
  - Build order payload
  - Submit to PesaPal API
  - Handle response (order_tracking_id, iframe URL)
  - Store transaction record

- [ ] **3.4** Implement IPN registration
  - Register IPN URL on app initialization
  - Handle IPN registration response
  - Store IPN ID in config/cache

---

### Phase 4: Controllers & Routes
- [ ] **4.1** Update `app/Http/Controllers/Api/PaymentController.php`
  - Add `initiatePesapal()` method
  - Add `pesapalCallback()` method
  - Add `pesapalIPN()` method
  - Keep existing M-Pesa methods

- [ ] **4.2** Create routes in `routes/api.php`
  ```php
  // PesaPal routes
  Route::post('/pesapal/initiate', [PaymentController::class, 'initiatePesapal']);
  Route::get('/pesapal/callback', [PaymentController::class, 'pesapalCallback']);
  Route::post('/pesapal/ipn', [PaymentController::class, 'pesapalIPN']);
  Route::get('/pesapal/status/{orderTrackingId}', [PaymentController::class, 'checkPesapalStatus']);
  ```

- [ ] **4.3** Add payment method selection logic
  - Update order creation to accept `payment_method` parameter
  - Route to appropriate payment gateway based on selection

---

### Phase 5: Order Integration
- [ ] **5.1** Update `Order` model
  - Add `payment_method` field to fillable array
  - Add relationship: `hasMany(PesapalTransaction::class)`
  - Add method to check if order has pending PesaPal transaction

- [ ] **5.2** Create migration to add `payment_method` to orders table
  ```php
  $table->enum('payment_method', ['mpesa', 'pesapal', 'cash'])->default('mpesa');
  ```

- [ ] **5.3** Update `OrderController`
  - Modify order creation flow
  - Add payment method validation
  - Route to PesaPal or M-Pesa based on selection

---

### Phase 6: Frontend Integration
- [ ] **6.1** Add payment method selection UI
  - Radio buttons or dropdown for M-Pesa vs PesaPal
  - Show appropriate form fields based on selection

- [ ] **6.2** Create PesaPal payment flow
  - Handle iframe rendering or redirect
  - Show loading/processing state
  - Handle callback after payment

- [ ] **6.3** Update order confirmation page
  - Show payment method used
  - Show transaction details
  - Handle both M-Pesa and PesaPal receipts

---

### Phase 7: Background Jobs & Queues
- [ ] **7.1** Create `ProcessPesapalCallback` job
  - Similar to `ProcessStkCallback`
  - Parse callback data
  - Update transaction status
  - Update order payment_status
  - Generate tickets if payment successful
  - Send confirmation SMS/email

- [ ] **7.2** Create `ProcessPesapalIPN` job
  - Handle Instant Payment Notifications
  - Update transaction records
  - Reconcile payment status

- [ ] **7.3** Add queue processing
  - Ensure queue worker is running
  - Test job dispatching

---

### Phase 8: Validation & Error Handling
- [ ] **8.1** Add request validation
  - Validate order details before PesaPal submission
  - Validate callback/IPN signatures
  - Validate transaction status responses

- [ ] **8.2** Implement error handling
  - Handle API failures gracefully
  - Log all PesaPal interactions
  - User-friendly error messages
  - Retry logic for failed requests

- [ ] **8.3** Add logging
  - Log all PesaPal API calls
  - Log callbacks and IPNs
  - Log transaction status changes

---

### Phase 9: Testing
- [ ] **9.1** Unit tests
  - Test PesapalService methods
  - Test order creation with PesaPal
  - Test transaction model methods

- [ ] **9.2** Integration tests
  - Test full payment flow with test credentials
  - Test callback handling
  - Test IPN handling
  - Test status checking

- [ ] **9.3** Manual testing with PesaPal sandbox
  - Complete test transaction
  - Verify callback processing
  - Verify IPN delivery
  - Test all payment methods (M-Pesa, Card, etc.)

- [ ] **9.4** Edge case testing
  - Failed payments
  - Cancelled payments
  - Duplicate callbacks
  - Network timeouts

---

### Phase 10: Security & Production Readiness
- [ ] **10.1** Implement security measures
  - Validate callback signatures
  - Verify IPN authenticity
  - Sanitize all inputs
  - Rate limiting on payment endpoints

- [ ] **10.2** Add transaction reconciliation
  - Daily reconciliation script
  - Compare local records with PesaPal
  - Flag discrepancies

- [ ] **10.3** Production configuration
  - Switch to production API keys
  - Update PESAPAL_BASE_URL to production
  - Configure production IPN/callback URLs
  - Test production credentials in staging

- [ ] **10.4** Documentation
  - Update API documentation
  - Create deployment guide
  - Document environment variables
  - Create troubleshooting guide

---

### Phase 11: Monitoring & Optimization
- [ ] **11.1** Add monitoring
  - Track payment success/failure rates
  - Monitor API response times
  - Alert on payment failures

- [ ] **11.2** Add analytics
  - Track payment method preferences
  - Track completion rates
  - Revenue by payment method

- [ ] **11.3** Performance optimization
  - Cache OAuth tokens
  - Optimize database queries
  - Add indexes where needed

---

## Key Implementation Notes

### PesaPal Flow:
1. User selects tickets and chooses PesaPal payment
2. Backend creates order and calls PesaPal API to submit order
3. PesaPal returns `order_tracking_id` and `iframe_url`
4. Frontend displays iframe or redirects user to payment page
5. User completes payment on PesaPal
6. PesaPal redirects user to callback URL
7. PesaPal sends IPN to registered IPN endpoint
8. Backend updates transaction and order status
9. Backend generates tickets and sends confirmation

### Payment Methods Supported by PesaPal:
- M-Pesa (Kenya, Tanzania, Uganda, Rwanda)
- Airtel Money
- Visa/Mastercard
- American Express
- Bank transfers
- Mobile wallets

### Critical Endpoints:
- **Submit Order**: `POST /api/Transactions/SubmitOrderRequest`
- **Get Transaction Status**: `GET /api/Transactions/GetTransactionStatus?orderTrackingId={id}`
- **Authentication**: `POST /api/Auth/RequestToken`
- **Register IPN**: `POST /api/URLSetup/RegisterIPN`

---

## Decision Points

### Choose Laravel Package vs Custom Implementation:
- **Option A**: Use `njoguamos/laravel-pesapal` package (Recommended)
  - Pros: Maintained, tested, documented
  - Cons: Additional dependency

- **Option B**: Use `knox/pesapal` package
  - Pros: Laravel 7+ compatible
  - Cons: Less recent updates

- **Option C**: Custom implementation
  - Pros: Full control, no dependencies
  - Cons: More development time, maintenance burden

**Decision**: [ ] Pending selection

### Multi-Currency Support:
- **Now**: Support KES only
- **Later**: Add support for UGX, TZS, RWF, etc.

**Decision**: [ ] Start with KES only

---

## Success Criteria
- [ ] Users can select PesaPal as payment method during checkout
- [ ] Payment iframe loads correctly
- [ ] Successful payments update order status to "paid"
- [ ] Tickets are generated automatically after successful payment
- [ ] Failed payments are properly logged and handled
- [ ] Users receive confirmation emails/SMS
- [ ] Admin can view all PesaPal transactions
- [ ] System handles callbacks and IPNs reliably
- [ ] Works in production with real transactions

---

## Timeline Estimates
_(To be filled based on team capacity)_

- Phase 1-2: Setup & Database - ___ days
- Phase 3-4: Core Implementation - ___ days
- Phase 5-6: Integration & UI - ___ days
- Phase 7-8: Jobs & Error Handling - ___ days
- Phase 9: Testing - ___ days
- Phase 10-11: Production & Monitoring - ___ days

**Total**: ___ days

---

## Additional Resources

### Official Documentation:
- PesaPal API v3 Docs: https://developer.pesapal.com/
- API Reference: https://developer.pesapal.com/api3/api-reference

### Community Resources:
- [Laravel PesaPal Package](https://github.com/njoguamos/laravel-pesapal)
- [Medium Guide](https://medium.com/@joshuawilfred/pesapal-payment-gateway-integration-with-rilt-stack-react-inertia-laravel-tailwindcss-8a8de8a71082)
- [Package by Steve Nyanümba](https://medium.com/@stevennmb9/how-i-built-a-laravel-integration-package-for-pesapals-latest-api-json-3-0-e2cea74592c2)

---

**Last Updated**: 2026-01-30
**Status**: Planning Phase
**Next Action**: Review and approve implementation approach
