<?php

namespace App\Filament\Pages;

use App\Services\SettingsService;
use Filament\Actions\Action;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Pages\Page;
use Filament\Notifications\Notification;
use Filament\Schemas\Schema;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;

class Settings extends Page implements HasForms
{
    use InteractsWithForms;

    protected static string|\BackedEnum|null $navigationIcon = 'heroicon-o-cog-6-tooth';
    protected static ?string $navigationLabel = 'System Settings';
    protected static \UnitEnum|string|null $navigationGroup = 'Configuration';
    protected static ?int $navigationSort = 10;

    protected string $view = 'filament.pages.settings';

    public ?array $data = [];

    public function mount(): void
    {
        $this->fillForm();
    }

    protected function getHeaderActions(): array
    {
        return [
            Action::make('save')
                ->label('Save Settings')
                ->action('save'),
        ];
    }

    protected function fillForm(): void
    {
        $settings = app(SettingsService::class);

        $this->data = [
            // Business
            'business_name' => $settings->get('business.name'),
            'business_email' => $settings->get('business.email'),
            'business_phone' => $settings->get('business.phone'),
            'business_address' => $settings->get('business.address'),
            'business_description' => $settings->get('business.description'),
            'business_logo_url' => $settings->get('business.logo_url'),
            'business_favicon_url' => $settings->get('business.favicon_url'),

            // SEO
            'seo_title' => $settings->get('seo.title'),
            'seo_description' => $settings->get('seo.description'),
            'seo_keywords' => $settings->get('seo.keywords'),

            // Social Media
            'social_facebook' => $settings->get('social.facebook'),
            'social_twitter' => $settings->get('social.twitter'),
            'social_instagram' => $settings->get('social.instagram'),
            'social_linkedin' => $settings->get('social.linkedin'),

            // Features
            'features_guest_checkout' => $settings->get('features.guest_checkout') === 'true',
            'features_mpesa_enabled' => $settings->get('features.mpesa_enabled') === 'true',
            'features_pesapal_enabled' => $settings->get('features.pesapal_enabled') === 'true',

            // Daraja/M-Pesa
            'daraja_base_url' => $settings->get('daraja.base_url'),
            'daraja_consumer_key' => $settings->get('daraja.key'),
            'daraja_consumer_secret' => $settings->get('daraja.secret'),
            'daraja_shortcode' => $settings->get('daraja.shortcode'),
            'daraja_passkey' => $settings->get('daraja.passkey'),
            'daraja_callback_url' => $settings->get('daraja.callback_url'),

            // Mail
            'mail_mailer' => $settings->get('mail.mailer'),
            'mail_host' => $settings->get('mail.host'),
            'mail_port' => $settings->get('mail.port'),
            'mail_username' => $settings->get('mail.username'),
            'mail_password' => $settings->get('mail.password'),
            'mail_encryption' => $settings->get('mail.encryption'),
            'mail_from_address' => $settings->get('mail.from.address'),

            // Africa's Talking
            'africastalking_username' => $settings->get('africastalking.username'),
            'africastalking_api_key' => $settings->get('africastalking.key'),
            'africastalking_from' => $settings->get('africastalking.from'),

            // PesaPal
            'pesapal_live' => $settings->get('pesapal.live') === 'true',
            'pesapal_consumer_key' => $settings->get('pesapal.consumer_key'),
            'pesapal_consumer_secret' => $settings->get('pesapal.consumer_secret'),
            'pesapal_callback_url' => $settings->get('pesapal.callback_url'),
            'pesapal_ipn_url' => $settings->get('pesapal.ipn_url'),
        ];
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->statePath('data')
            ->components([
                Tabs::make('Settings')
                    ->tabs([
                        Tab::make('Business')
                            ->icon('heroicon-o-building-storefront')
                            ->schema([
                                TextInput::make('business_name')
                                    ->label('Business Name')
                                    ->required()
                                    ->helperText('This name appears throughout the application including admin panel and frontend'),

                                TextInput::make('business_email')
                                    ->label('Contact Email')
                                    ->email()
                                    ->required(),

                                TextInput::make('business_phone')
                                    ->label('Contact Phone')
                                    ->tel()
                                    ->required(),

                                TextInput::make('business_address')
                                    ->label('Address')
                                    ->columnSpanFull(),

                                Textarea::make('business_description')
                                    ->label('Description')
                                    ->rows(3)
                                    ->columnSpanFull()
                                    ->helperText('Brief description of your business'),

                                TextInput::make('business_logo_url')
                                    ->label('Logo URL')
                                    ->url()
                                    ->helperText('Full URL to your logo image'),

                                TextInput::make('business_favicon_url')
                                    ->label('Favicon URL')
                                    ->url()
                                    ->helperText('Full URL to your favicon image'),
                            ])
                            ->columns(2),

                        Tab::make('SEO')
                            ->icon('heroicon-o-magnifying-glass')
                            ->schema([
                                TextInput::make('seo_title')
                                    ->label('SEO Title')
                                    ->required()
                                    ->maxLength(60)
                                    ->helperText('Recommended: 50-60 characters'),

                                Textarea::make('seo_description')
                                    ->label('SEO Description')
                                    ->required()
                                    ->rows(3)
                                    ->maxLength(160)
                                    ->helperText('Recommended: 150-160 characters')
                                    ->columnSpanFull(),

                                Textarea::make('seo_keywords')
                                    ->label('Keywords')
                                    ->rows(2)
                                    ->helperText('Comma-separated keywords for search engines')
                                    ->columnSpanFull(),
                            ])
                            ->columns(2),

                        Tab::make('Social Media')
                            ->icon('heroicon-o-share')
                            ->schema([
                                TextInput::make('social_facebook')
                                    ->label('Facebook URL')
                                    ->url()
                                    ->placeholder('https://facebook.com/yourpage'),

                                TextInput::make('social_twitter')
                                    ->label('Twitter/X URL')
                                    ->url()
                                    ->placeholder('https://twitter.com/yourhandle'),

                                TextInput::make('social_instagram')
                                    ->label('Instagram URL')
                                    ->url()
                                    ->placeholder('https://instagram.com/yourprofile'),

                                TextInput::make('social_linkedin')
                                    ->label('LinkedIn URL')
                                    ->url()
                                    ->placeholder('https://linkedin.com/company/yourcompany'),
                            ])
                            ->columns(2),

                        Tab::make('Features')
                            ->icon('heroicon-o-cog-6-tooth')
                            ->schema([
                                Toggle::make('features_guest_checkout')
                                    ->label('Enable Guest Checkout')
                                    ->helperText('Allow customers to purchase without creating an account'),

                                Toggle::make('features_mpesa_enabled')
                                    ->label('Enable M-Pesa Payments')
                                    ->helperText('Accept payments via M-Pesa'),

                                Toggle::make('features_pesapal_enabled')
                                    ->label('Enable PesaPal Payments')
                                    ->helperText('Accept payments via PesaPal (Cards & Mobile Money)'),
                            ])
                            ->columns(1),

                        Tab::make('Payment (M-Pesa)')
                            ->icon('heroicon-o-credit-card')
                            ->schema([
                                TextInput::make('daraja_base_url')
                                    ->label('Daraja API Base URL')
                                    ->required()
                                    ->default('https://sandbox.safaricom.co.ke')
                                    ->url()
                                    ->helperText('Use sandbox URL for testing, production URL for live'),

                                TextInput::make('daraja_consumer_key')
                                    ->label('Consumer Key')
                                    ->required()
                                    ->password()
                                    ->revealable(),

                                TextInput::make('daraja_consumer_secret')
                                    ->label('Consumer Secret')
                                    ->required()
                                    ->password()
                                    ->revealable(),

                                TextInput::make('daraja_shortcode')
                                    ->label('Business Shortcode')
                                    ->required()
                                    ->numeric(),

                                TextInput::make('daraja_passkey')
                                    ->label('Lipa Na M-Pesa Passkey')
                                    ->required()
                                    ->password()
                                    ->revealable(),

                                TextInput::make('daraja_callback_url')
                                    ->label('Callback URL')
                                    ->required()
                                    ->url()
                                    ->helperText('URL where payment confirmations will be sent'),
                            ])
                            ->columns(2),

                        Tab::make('Payment (PesaPal)')
                            ->icon('heroicon-o-banknotes')
                            ->schema([
                                Toggle::make('pesapal_live')
                                    ->label('Enable Live Mode')
                                    ->helperText('Toggle between sandbox (testing) and live (production) mode')
                                    ->columnSpanFull(),

                                TextInput::make('pesapal_consumer_key')
                                    ->label('Consumer Key')
                                    ->required()
                                    ->password()
                                    ->revealable()
                                    ->helperText('PesaPal API Consumer Key'),

                                TextInput::make('pesapal_consumer_secret')
                                    ->label('Consumer Secret')
                                    ->required()
                                    ->password()
                                    ->revealable()
                                    ->helperText('PesaPal API Consumer Secret'),

                                TextInput::make('pesapal_callback_url')
                                    ->label('Callback URL')
                                    ->required()
                                    ->url()
                                    ->helperText('URL where users are redirected after payment')
                                    ->placeholder(url('/api/v1/pesapal/callback')),

                                TextInput::make('pesapal_ipn_url')
                                    ->label('IPN URL')
                                    ->required()
                                    ->url()
                                    ->helperText('Instant Payment Notification URL for payment status updates')
                                    ->placeholder(url('/api/v1/pesapal/ipn')),
                            ])
                            ->columns(2),

                        Tab::make('Email')
                            ->icon('heroicon-o-envelope')
                            ->schema([
                                TextInput::make('mail_mailer')
                                    ->label('Mail Driver')
                                    ->default('smtp')
                                    ->required(),

                                TextInput::make('mail_host')
                                    ->label('SMTP Host')
                                    ->required(),

                                TextInput::make('mail_port')
                                    ->label('SMTP Port')
                                    ->numeric()
                                    ->default(587)
                                    ->required(),

                                TextInput::make('mail_username')
                                    ->label('SMTP Username')
                                    ->required(),

                                TextInput::make('mail_password')
                                    ->label('SMTP Password')
                                    ->password()
                                    ->revealable()
                                    ->required(),

                                TextInput::make('mail_encryption')
                                    ->label('Encryption')
                                    ->default('tls'),

                                TextInput::make('mail_from_address')
                                    ->label('From Email Address')
                                    ->email()
                                    ->required()
                                    ->helperText('Email address that appears as sender'),
                            ])
                            ->columns(2),

                        Tab::make('SMS')
                            ->icon('heroicon-o-chat-bubble-left-right')
                            ->schema([
                                Placeholder::make('sms_info')
                                    ->content('Configure Africa\'s Talking API for SMS notifications')
                                    ->columnSpanFull(),

                                TextInput::make('africastalking_username')
                                    ->label('API Username')
                                    ->required(),

                                TextInput::make('africastalking_api_key')
                                    ->label('API Key')
                                    ->password()
                                    ->revealable()
                                    ->required(),

                                TextInput::make('africastalking_from')
                                    ->label('Sender ID')
                                    ->helperText('Approved sender name (alphanumeric)')
                                    ->required(),
                            ])
                            ->columns(2),
                    ])
                    ->columnSpanFull(),
            ]);
    }

    public function save(): void
    {
        $data = $this->data;
        $settings = app(SettingsService::class);

        // Save business settings
        if (isset($data['business_name'])) {
            $settings->set('business.name', $data['business_name'], 'business', false);
        }
        if (isset($data['business_email'])) {
            $settings->set('business.email', $data['business_email'], 'business', false);
        }
        if (isset($data['business_phone'])) {
            $settings->set('business.phone', $data['business_phone'], 'business', false);
        }
        if (isset($data['business_address'])) {
            $settings->set('business.address', $data['business_address'], 'business', false);
        }
        if (isset($data['business_description'])) {
            $settings->set('business.description', $data['business_description'], 'business', false);
        }
        if (isset($data['business_logo_url'])) {
            $settings->set('business.logo_url', $data['business_logo_url'], 'business', false);
        }
        if (isset($data['business_favicon_url'])) {
            $settings->set('business.favicon_url', $data['business_favicon_url'], 'business', false);
        }

        // Save SEO settings
        if (isset($data['seo_title'])) {
            $settings->set('seo.title', $data['seo_title'], 'seo', false);
        }
        if (isset($data['seo_description'])) {
            $settings->set('seo.description', $data['seo_description'], 'seo', false);
        }
        if (isset($data['seo_keywords'])) {
            $settings->set('seo.keywords', $data['seo_keywords'], 'seo', false);
        }

        // Save social media settings
        if (isset($data['social_facebook'])) {
            $settings->set('social.facebook', $data['social_facebook'], 'social', false);
        }
        if (isset($data['social_twitter'])) {
            $settings->set('social.twitter', $data['social_twitter'], 'social', false);
        }
        if (isset($data['social_instagram'])) {
            $settings->set('social.instagram', $data['social_instagram'], 'social', false);
        }
        if (isset($data['social_linkedin'])) {
            $settings->set('social.linkedin', $data['social_linkedin'], 'social', false);
        }

        // Save feature flags
        $settings->set('features.guest_checkout', $data['features_guest_checkout'] ? 'true' : 'false', 'features', false);
        $settings->set('features.mpesa_enabled', $data['features_mpesa_enabled'] ? 'true' : 'false', 'features', false);
        $settings->set('features.pesapal_enabled', $data['features_pesapal_enabled'] ? 'true' : 'false', 'features', false);

        // Save Daraja settings (encrypted)
        if (isset($data['daraja_base_url'])) {
            $settings->set('daraja.base_url', $data['daraja_base_url'], 'payment', false);
        }
        if (isset($data['daraja_consumer_key'])) {
            $settings->set('daraja.key', $data['daraja_consumer_key'], 'payment', true);
        }
        if (isset($data['daraja_consumer_secret'])) {
            $settings->set('daraja.secret', $data['daraja_consumer_secret'], 'payment', true);
        }
        if (isset($data['daraja_shortcode'])) {
            $settings->set('daraja.shortcode', $data['daraja_shortcode'], 'payment', false);
        }
        if (isset($data['daraja_passkey'])) {
            $settings->set('daraja.passkey', $data['daraja_passkey'], 'payment', true);
        }
        if (isset($data['daraja_callback_url'])) {
            $settings->set('daraja.callback_url', $data['daraja_callback_url'], 'payment', false);
        }

        // Save mail settings
        if (isset($data['mail_mailer'])) {
            $settings->set('mail.mailer', $data['mail_mailer'], 'mail', false);
        }
        if (isset($data['mail_host'])) {
            $settings->set('mail.host', $data['mail_host'], 'mail', false);
        }
        if (isset($data['mail_port'])) {
            $settings->set('mail.port', $data['mail_port'], 'mail', false);
        }
        if (isset($data['mail_username'])) {
            $settings->set('mail.username', $data['mail_username'], 'mail', false);
        }
        if (isset($data['mail_password'])) {
            $settings->set('mail.password', $data['mail_password'], 'mail', true);
        }
        if (isset($data['mail_encryption'])) {
            $settings->set('mail.encryption', $data['mail_encryption'], 'mail', false);
        }
        if (isset($data['mail_from_address'])) {
            $settings->set('mail.from.address', $data['mail_from_address'], 'mail', false);
        }

        // Save SMS settings
        if (isset($data['africastalking_username'])) {
            $settings->set('africastalking.username', $data['africastalking_username'], 'sms', false);
        }
        if (isset($data['africastalking_api_key'])) {
            $settings->set('africastalking.key', $data['africastalking_api_key'], 'sms', true);
        }
        if (isset($data['africastalking_from'])) {
            $settings->set('africastalking.from', $data['africastalking_from'], 'sms', false);
        }

        // Save PesaPal settings
        $settings->set('pesapal.live', $data['pesapal_live'] ? 'true' : 'false', 'payment', false);
        if (isset($data['pesapal_consumer_key'])) {
            $settings->set('pesapal.consumer_key', $data['pesapal_consumer_key'], 'payment', true);
        }
        if (isset($data['pesapal_consumer_secret'])) {
            $settings->set('pesapal.consumer_secret', $data['pesapal_consumer_secret'], 'payment', true);
        }
        if (isset($data['pesapal_callback_url'])) {
            $settings->set('pesapal.callback_url', $data['pesapal_callback_url'], 'payment', false);
        }
        if (isset($data['pesapal_ipn_url'])) {
            $settings->set('pesapal.ipn_url', $data['pesapal_ipn_url'], 'payment', false);
        }

        // Clear settings cache to reflect changes immediately
        $settings->flush();

        Notification::make()
            ->success()
            ->title('Settings saved successfully!')
            ->body('All settings have been updated and will take effect immediately.')
            ->send();
    }

    public static function canAccess(): bool
    {
        return auth()->user()->hasRole(['super_admin', 'admin']);
    }
}
