<?php

namespace App\Filament\Resources\Agents;

use App\Filament\Resources\Agents\Pages\CreateAgent;
use App\Filament\Resources\Agents\Pages\EditAgent;
use App\Filament\Resources\Agents\Pages\ListAgents;
use App\Filament\Resources\Agents\Pages\ViewAgent;
use App\Models\Agent;
use App\Models\Event;
use App\Models\User;
use Filament\Forms\Components\Radio;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Get;
use Filament\Schemas\Schema;
use Filament\Resources\Resource;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;

class AgentResource extends Resource
{
    protected static ?string $model = Agent::class;

    protected static string|\BackedEnum|null $navigationIcon = 'heroicon-o-user-group';

    protected static string|\UnitEnum|null $navigationGroup = 'Users';

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Radio::make('user_mode')
                    ->label('User Selection Mode')
                    ->options([
                        'new' => 'Create New User',
                        'existing' => 'Select Existing User',
                    ])
                    ->default('new')
                    ->required()
                    ->live()
                    ->hiddenOn('edit'),

                Select::make('user_id')
                    ->label('Select User')
                    ->options(function () {
                        // Get users who are not already agents
                        return User::whereDoesntHave('agent')
                            ->pluck('name', 'id');
                    })
                    ->searchable()
                    ->required()
                    ->visible(fn (Get $get): bool => $get('user_mode') === 'existing')
                    ->hiddenOn('edit'),

                TextInput::make('name')
                    ->label('Agent Name')
                    ->required()
                    ->maxLength(255)
                    ->visible(fn (Get $get): bool => $get('user_mode') === 'new' || request()->routeIs('*.edit')),

                TextInput::make('phone')
                    ->label('Phone Number')
                    ->required()
                    ->maxLength(255)
                    ->visible(fn (Get $get): bool => $get('user_mode') === 'new' || request()->routeIs('*.edit')),

                TextInput::make('email')
                    ->label('Email Address')
                    ->email()
                    ->required()
                    ->visible(fn (Get $get): bool => $get('user_mode') === 'new' || request()->routeIs('*.edit'))
                    ->rules([
                        function () {
                            return function (string $attribute, $value, \Closure $fail) {
                                $agent = request()->route('record');
                                $userId = $agent ? $agent->user_id : null;

                                $exists = User::where('email', $value)
                                    ->when($userId, fn($query) => $query->where('id', '!=', $userId))
                                    ->exists();

                                if ($exists) {
                                    $fail('The email address is already taken.');
                                }
                            };
                        },
                    ]),

                TextInput::make('password')
                    ->label('Password')
                    ->password()
                    ->minLength(8)
                    ->dehydrated(fn ($state) => filled($state))
                    ->visible(fn (Get $get): bool => $get('user_mode') === 'new')
                    ->hiddenOn('edit')
                    ->helperText('Leave empty to use default password: password123'),

                Repeater::make('assigned_events')
                    ->label('Assigned Events')
                    ->schema([
                        Select::make('event_id')
                            ->label('Event')
                            ->options(Event::pluck('name', 'id'))
                            ->searchable()
                            ->required()
                            ->distinct()
                            ->disableOptionsWhenSelectedInSiblingRepeaterItems(),
                    ])
                    ->columnSpanFull()
                    ->defaultItems(0)
                    ->reorderable()
                    ->collapsible()
                    ->itemLabel(fn (array $state): ?string =>
                        Event::find($state['event_id'])?->name ?? 'Select an event'
                    )
                    ->addActionLabel('Add Event')
                    ->afterStateHydrated(function ($component, $state) {
                        // Convert array of IDs from database to repeater format
                        if (is_array($state) && !empty($state)) {
                            // Check if it's already in repeater format
                            $firstItem = reset($state);
                            if (!is_array($firstItem)) {
                                // Convert [1, 2, 3] to [['event_id' => 1], ['event_id' => 2], ['event_id' => 3]]
                                $component->state(
                                    collect($state)->map(fn ($id) => ['event_id' => $id])->toArray()
                                );
                            }
                        }
                    })
                    ->dehydrateStateUsing(function ($state) {
                        // Convert repeater format to array of IDs for database storage
                        // [['event_id' => 1], ['event_id' => 2]] to [1, 2, 3]
                        return collect($state)->pluck('event_id')->filter()->values()->toArray();
                    }),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('user.name')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('user.email')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('assigned_events')
                    ->label('Assigned Events')
                    ->badge()
                    ->getStateUsing(function ($record) {
                        if (empty($record->assigned_events)) {
                            return ['None'];
                        }
                        return Event::whereIn('id', $record->assigned_events)->pluck('name')->toArray();
                    })
                    ->separator(',')
                    ->wrap(),
                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->recordActions([
                \Filament\Actions\ViewAction::make(),
                \Filament\Actions\EditAction::make(),
            ])
            ->bulkActions([
                \Filament\Actions\BulkActionGroup::make([
                    \Filament\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListAgents::route('/'),
            'create' => CreateAgent::route('/create'),
            'view' => ViewAgent::route('/{record}'),
            'edit' => EditAgent::route('/{record}/edit'),
        ];
    }
}
