<?php

namespace App\Filament\Resources\Events\RelationManagers;

use Filament\Forms;
use Filament\Schemas\Schema;
use Filament\Schemas\Components\Grid;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\CreateAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Tables\Table;

class CategoriesRelationManager extends RelationManager
{
    protected static string $relationship = 'categories';

    protected static ?string $recordTitleAttribute = 'name';

    protected static ?string $title = 'Ticket Categories';

    public function form(Schema $schema): Schema
    {
        return $schema
            ->schema([
                Grid::make(2)
                    ->schema([
                        Forms\Components\TextInput::make('name')
                            ->required()
                            ->maxLength(255)
                            ->placeholder('e.g., VIP, General Admission, Early Bird')
                            ->helperText('Enter the ticket category name')
                            ->columnSpan(2),

                        Forms\Components\TextInput::make('price')
                            ->required()
                            ->numeric()
                            ->prefix('KES')
                            ->minValue(0)
                            ->placeholder('0.00')
                            ->helperText('Enter ticket price in Kenyan Shillings'),

                        Forms\Components\TextInput::make('quantity')
                            ->required()
                            ->numeric()
                            ->minValue(1)
                            ->placeholder('100')
                            ->helperText('Total tickets available for this category'),
                    ]),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->searchable()
                    ->sortable()
                    ->weight('bold'),

                Tables\Columns\TextColumn::make('price')
                    ->money('KES', divideBy: 1)
                    ->sortable(),

                Tables\Columns\TextColumn::make('quantity')
                    ->label('Total Available')
                    ->sortable()
                    ->alignCenter(),

                Tables\Columns\TextColumn::make('tickets_count')
                    ->label('Sold')
                    ->counts('tickets')
                    ->sortable()
                    ->alignCenter(),

                Tables\Columns\TextColumn::make('remaining')
                    ->label('Remaining')
                    ->getStateUsing(fn ($record) => $record->quantity - $record->tickets_count)
                    ->badge()
                    ->color(fn ($state) => match (true) {
                        $state > 50 => 'success',
                        $state > 10 => 'warning',
                        $state > 0 => 'danger',
                        default => 'gray',
                    })
                    ->alignCenter(),

                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                CreateAction::make()
                    ->modalHeading('Add Ticket Category')
                    ->modalDescription('Create a new ticket category for this event')
                    ->modalWidth('lg')
                    ->createAnother(false)
                    ->successNotificationTitle('Category created'),
            ])
            ->recordActions([
                EditAction::make()
                    ->modalHeading('Edit Ticket Category')
                    ->modalWidth('lg')
                    ->successNotificationTitle('Category updated'),

                DeleteAction::make()
                    ->requiresConfirmation()
                    ->modalHeading('Delete Ticket Category')
                    ->modalDescription('Are you sure you want to delete this category? All associated tickets will also be deleted.')
                    ->successNotificationTitle('Category deleted'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ])
            ->emptyStateHeading('No ticket categories yet')
            ->emptyStateDescription('Create your first ticket category to start selling tickets for this event.')
            ->emptyStateIcon('heroicon-o-ticket');
    }
}
