<?php

namespace App\Filament\Resources\Events\Schemas;

use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\TimePicker;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Repeater;
use Filament\Schemas\Schema;

class EventForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                TextInput::make('name')
                    ->required(),
                Textarea::make('description')
                    ->columnSpanFull(),
                FileUpload::make('image_path')
                    ->label('Event Image')
                    ->image()
                    ->directory('events')
                    ->disk('public')
                    ->imageEditor()
                    ->maxSize(5120)
                    ->helperText('Upload an event image (max 5MB). Recommended size: 1200x675px')
                    ->columnSpanFull(),
                TextInput::make('venue')
                    ->required(),
                DatePicker::make('event_date')
                    ->required(),
                TimePicker::make('start_time')
                    ->required(),
                TimePicker::make('end_time')
                    ->required(),
                TextInput::make('max_capacity')
                    ->label('Maximum Capacity')
                    ->numeric()
                    ->minValue(1)
                    ->placeholder('e.g., 1000')
                    ->helperText('Total maximum attendees for this event (optional)')
                    ->nullable(),
                Toggle::make('enforce_capacity')
                    ->label('Enforce Capacity Limit')
                    ->helperText('Prevent ticket sales when capacity is reached')
                    ->default(true)
                    ->visible(fn ($get) => $get('max_capacity') !== null),
                Select::make('status')
                    ->options([
                        'upcoming' => 'Upcoming',
                        'ongoing' => 'Ongoing',
                        'completed' => 'Completed',
                        'cancelled' => 'Cancelled',
                    ])
                    ->default('upcoming')
                    ->required(),
                Select::make('category')
                    ->options([
                        'Music & Concerts' => 'Music & Concerts',
                        'Sports & Fitness' => 'Sports & Fitness',
                        'Arts & Theater' => 'Arts & Theater',
                        'Business & Corporate' => 'Business & Corporate',
                        'Food & Drink' => 'Food & Drink',
                        'Community & Culture' => 'Community & Culture',
                        'Educational' => 'Educational',
                        'Other' => 'Other',
                    ])
                    ->searchable()
                    ->helperText('Select the event category'),
                Toggle::make('featured')
                    ->label('Feature this event')
                    ->helperText('Featured events appear in the homepage carousel')
                    ->default(false),
                Select::make('created_by')
                    ->relationship('creator', 'name')
                    ->required()
                    ->default(fn() => auth()->id()),
                Repeater::make('categories')
                    ->relationship('categories')
                    ->schema([
                        TextInput::make('name')
                            ->label('Category Name')
                            ->required()
                            ->placeholder('e.g., Kids, Adults, VIP, Student')
                            ->helperText('Examples: Kids (5-12 years), Adults, VIP, Early Bird, Student Pass'),
                        TextInput::make('price')
                            ->label('Price')
                            ->required()
                            ->numeric()
                            ->prefix('KES')
                            ->placeholder('e.g., 1500')
                            ->minValue(0),
                        TextInput::make('quantity')
                            ->label('Available Tickets')
                            ->required()
                            ->numeric()
                            ->minValue(1)
                            ->placeholder('e.g., 100')
                            ->helperText('Total number of tickets for this category'),
                    ])
                    ->columns(3)
                    ->defaultItems(0)
                    ->addActionLabel('Add Ticket Category')
                    ->collapsible()
                    ->itemLabel(fn (array $state): ?string => $state['name'] ?? null)
                    ->columnSpanFull()
                    ->orderColumn(false)
                    ->helperText('Add different ticket categories for your event (e.g., Kids, Adults, VIP)'),
            ]);
    }
}
