<?php

namespace App\Filament\Resources\Menus;

use App\Filament\Resources\Menus\Pages\CreateMenu;
use App\Filament\Resources\Menus\Pages\EditMenu;
use App\Filament\Resources\Menus\Pages\ListMenus;
use App\Models\Menu;
use Filament\Actions\EditAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Resources\Resource;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Repeater;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;

class MenuResource extends Resource
{
    protected static ?string $model = Menu::class;
    protected static string|\BackedEnum|null $navigationIcon = 'heroicon-o-bars-3';
    protected static \UnitEnum|string|null $navigationGroup = 'Content Management';
    protected static ?int $navigationSort = 2;

    public static function form(Schema $schema): Schema
    {
        return $schema->components([
            Section::make('Menu Details')->schema([
                TextInput::make('name')
                    ->required()
                    ->maxLength(255),

                Select::make('location')
                    ->options([
                        'topbar' => 'Top Navigation',
                        'footer' => 'Footer',
                    ])
                    ->required()
                    ->unique(ignoreRecord: true),

                Select::make('status')
                    ->options([
                        'active' => 'Active',
                        'inactive' => 'Inactive',
                    ])
                    ->default('active')
                    ->required(),
            ])->columns(3),

            Section::make('Menu Items')->schema([
                Repeater::make('items')
                    ->relationship('items')
                    ->schema([
                        TextInput::make('label')
                            ->required()
                            ->columnSpan(2),

                        Select::make('url_type')
                            ->options([
                                'custom' => 'Custom URL',
                                'page' => 'CMS Page',
                                'route' => 'Named Route',
                            ])
                            ->default('custom')
                            ->live()
                            ->required(),

                        TextInput::make('url')
                            ->label('URL')
                            ->visible(fn ($get) => $get('url_type') === 'custom')
                            ->required(fn ($get) => $get('url_type') === 'custom')
                            ->columnSpan(2),

                        Select::make('page_id')
                            ->relationship('page', 'title')
                            ->visible(fn ($get) => $get('url_type') === 'page')
                            ->required(fn ($get) => $get('url_type') === 'page')
                            ->columnSpan(2),

                        Select::make('target')
                            ->options([
                                '_self' => 'Same Window',
                                '_blank' => 'New Window',
                            ])
                            ->default('_self'),

                        TextInput::make('order')
                            ->numeric()
                            ->default(0),

                        // Nested repeater for child menu items
                        Repeater::make('children')
                            ->relationship('children')
                            ->schema([
                                TextInput::make('label')
                                    ->required(),

                                Select::make('url_type')
                                    ->options([
                                        'custom' => 'Custom URL',
                                        'page' => 'CMS Page',
                                        'route' => 'Named Route',
                                    ])
                                    ->default('custom')
                                    ->live()
                                    ->required(),

                                TextInput::make('url')
                                    ->label('URL')
                                    ->visible(fn ($get) => $get('url_type') === 'custom')
                                    ->required(fn ($get) => $get('url_type') === 'custom'),

                                Select::make('page_id')
                                    ->relationship('page', 'title')
                                    ->visible(fn ($get) => $get('url_type') === 'page')
                                    ->required(fn ($get) => $get('url_type') === 'page'),

                                Select::make('target')
                                    ->options([
                                        '_self' => 'Same Window',
                                        '_blank' => 'New Window',
                                    ])
                                    ->default('_self'),

                                TextInput::make('order')
                                    ->numeric()
                                    ->default(0),
                            ])
                            ->mutateRelationshipDataBeforeCreateUsing(function (array $data, $record, $livewire): array {
                                $data['menu_id'] = $livewire->record->id;
                                return $data;
                            })
                            ->mutateRelationshipDataBeforeFillUsing(function (array $data): array {
                                return $data;
                            })
                            ->columns(3)
                            ->defaultItems(0)
                            ->itemLabel(fn (array $state): ?string => $state['label'] ?? null)
                            ->reorderable('order')
                            ->collapsible()
                            ->columnSpanFull()
                            ->label('Child Links (for Footer sections)'),
                    ])
                    ->columns(3)
                    ->defaultItems(0)
                    ->itemLabel(fn (array $state): ?string => $state['label'] ?? null)
                    ->reorderable('order')
                    ->collapsible(),
            ]),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')->searchable()->sortable(),
                TextColumn::make('location')->badge(),
                TextColumn::make('status')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'active' => 'success',
                        'inactive' => 'danger',
                        default => 'gray',
                    }),
                TextColumn::make('items_count')
                    ->counts('items')
                    ->label('Items'),
            ])
            ->recordActions([
                EditAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => ListMenus::route('/'),
            'create' => CreateMenu::route('/create'),
            'edit' => EditMenu::route('/{record}/edit'),
        ];
    }

    public static function canViewAny(): bool
    {
        return auth()->user()->hasRole(['super_admin', 'admin']);
    }
}
