<?php

namespace App\Filament\Resources\Pages\Schemas;

use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\KeyValue;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Illuminate\Support\Str;

class PageForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            Section::make('Content')
                ->schema([
                    TextInput::make('title')
                        ->required()
                        ->maxLength(255)
                        ->live(onBlur: true)
                        ->afterStateUpdated(fn ($state, callable $set) =>
                            $set('slug', Str::slug($state))
                        ),

                    TextInput::make('slug')
                        ->required()
                        ->unique(ignoreRecord: true)
                        ->maxLength(255)
                        ->helperText('URL-friendly version of the title'),

                    Textarea::make('excerpt')
                        ->rows(3)
                        ->helperText('Brief summary for previews'),

                    RichEditor::make('content')
                        ->required()
                        ->toolbarButtons([
                            'bold', 'italic', 'underline', 'strike',
                            'h2', 'h3', 'bulletList', 'orderedList',
                            'link', 'undo', 'redo',
                        ])
                        ->columnSpanFull(),
                ])->columns(2),

            Section::make('Settings')
                ->schema([
                    Select::make('status')
                        ->options([
                            'draft' => 'Draft',
                            'published' => 'Published',
                        ])
                        ->default('draft')
                        ->required(),

                    Select::make('template')
                        ->options([
                            'default' => 'Default',
                            'full-width' => 'Full Width',
                        ])
                        ->default('default'),

                    Select::make('created_by')
                        ->relationship('creator', 'name')
                        ->default(fn() => auth()->id())
                        ->required()
                        ->disabled(),
                ])->columns(3),

            Section::make('SEO')
                ->schema([
                    KeyValue::make('meta')
                        ->keyLabel('Meta Key')
                        ->valueLabel('Meta Value')
                        ->addActionLabel('Add Meta Tag')
                        ->helperText('Add custom meta tags for SEO')
                ])->collapsible(),
        ]);
    }
}
