<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Agent;
use Illuminate\Http\Request;

class AgentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return Agent::with('user')->paginate(20);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Support two modes:
        // 1. Create agent with existing user_id
        // 2. Create agent with new user details (name, email, phone, password)

        $userId = null;

        if ($request->has('user_id')) {
            // Mode 1: Use existing user
            $validated = $request->validate([
                'user_id' => 'required|exists:users,id|unique:agents,user_id',
                'assigned_events' => 'nullable|array',
            ]);

            $userId = $validated['user_id'];
        } else {
            // Mode 2: Create new user
            $userValidated = $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|unique:users,email',
                'phone' => 'required|string|max:255',
                'password' => 'nullable|string|min:8',
                'assigned_events' => 'nullable|array',
            ]);

            // Create the user
            $user = \App\Models\User::create([
                'name' => $userValidated['name'],
                'email' => $userValidated['email'],
                'phone' => $userValidated['phone'],
                'password' => bcrypt($userValidated['password'] ?? 'password123'),
                'role' => 'agent',
                'status' => 'active',
            ]);

            $userId = $user->id;
            $validated = ['assigned_events' => $userValidated['assigned_events'] ?? []];
        }

        // Create the agent
        $agent = Agent::create([
            'user_id' => $userId,
            'assigned_events' => $validated['assigned_events'] ?? [],
        ]);

        // Ensure user has agent role
        $user = \App\Models\User::find($userId);
        if (!$user->hasRole('agent')) {
            $user->assignRole('agent');
        }

        return response()->json($agent->load('user'), 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        return Agent::with('user')->findOrFail($id);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $agent = Agent::findOrFail($id);

        $validated = $request->validate([
            'assigned_events' => 'nullable|array',
        ]);

        $agent->update($validated);

        return response()->json($agent);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $agent = Agent::findOrFail($id);
        
        // Note: we might want to remove the role from the user too, but maybe not.
        
        $agent->delete();

        return response()->json(null, 204);
    }
}
