<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Menu;
use Illuminate\Http\JsonResponse;

class MenuController extends Controller
{
    /**
     * Get menu by location
     *
     * @param string $location
     * @return JsonResponse
     */
    public function getByLocation(string $location): JsonResponse
    {
        $menu = Menu::with(['items' => function ($query) {
            $query->whereNull('parent_id')
                  ->with(['children' => function ($childQuery) {
                      $childQuery->orderBy('order');
                  }])
                  ->orderBy('order');
        }, 'items.page'])
        ->where('location', $location)
        ->where('status', 'active')
        ->first();

        if (!$menu) {
            return response()->json([
                'success' => false,
                'message' => 'Menu not found',
            ], 404);
        }

        // Transform items to include resolved URLs
        $transformedItems = $menu->items->map(function ($item) {
            return [
                'id' => $item->id,
                'label' => $item->label,
                'url' => $item->resolved_url,
                'target' => $item->target,
                'children' => $item->children->map(function ($child) {
                    return [
                        'id' => $child->id,
                        'label' => $child->label,
                        'url' => $child->resolved_url,
                        'target' => $child->target,
                    ];
                }),
            ];
        });

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $menu->id,
                'name' => $menu->name,
                'location' => $menu->location,
                'items' => $transformedItems,
            ],
        ]);
    }

    /**
     * Get all active menus
     *
     * @return JsonResponse
     */
    public function index(): JsonResponse
    {
        $menus = Menu::with(['items' => function ($query) {
            $query->whereNull('parent_id')
                  ->with(['children' => function ($childQuery) {
                      $childQuery->orderBy('order');
                  }])
                  ->orderBy('order');
        }, 'items.page'])
        ->where('status', 'active')
        ->get();

        $transformedMenus = $menus->map(function ($menu) {
            $transformedItems = $menu->items->map(function ($item) {
                return [
                    'id' => $item->id,
                    'label' => $item->label,
                    'url' => $item->resolved_url,
                    'target' => $item->target,
                    'children' => $item->children->map(function ($child) {
                        return [
                            'id' => $child->id,
                            'label' => $child->label,
                            'url' => $child->resolved_url,
                            'target' => $child->target,
                        ];
                    }),
                ];
            });

            return [
                'id' => $menu->id,
                'name' => $menu->name,
                'location' => $menu->location,
                'items' => $transformedItems,
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $transformedMenus,
        ]);
    }
}
