<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\TicketCategory;
use App\Services\DarajaService;
use App\Services\TicketHoldService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class OrderController extends Controller
{
    protected $daraja;
    protected $ticketHoldService;

    public function __construct(DarajaService $daraja, TicketHoldService $ticketHoldService)
    {
        $this->daraja = $daraja;
        $this->ticketHoldService = $ticketHoldService;
    }

    /**
     * Store a new order and initiate payment.
     */
    public function store(Request $request)
    {
        $request->validate([
            'event_id' => 'required|exists:events,id',
            'ticket_category_id' => 'required|exists:ticket_categories,id',
            'phone_number' => 'required|string',
            'quantity' => 'nullable|integer|min:1|max:10',
            'email' => 'required_without:user_id|email',
            'full_name' => 'required_without:user_id|string',
        ]);

        $quantity = $request->quantity ?? 1;

        // Get or create user (support both authenticated and guest users)
        $user = $request->user() ?? $this->findOrCreateGuestUser($request);

        $category = TicketCategory::findOrFail($request->ticket_category_id);

        // Verify category belongs to event
        if ($category->event_id != $request->event_id) {
            return response()->json(['message' => 'Invalid ticket category for this event'], 422);
        }

        // Check category availability (accounting for holds)
        $available = $this->ticketHoldService->getAvailableQuantity($category->id);

        if ($available <= 0) {
             return response()->json(['message' => 'Tickets sold out for this category'], 422);
        }

        if ($quantity > $available) {
             return response()->json([
                 'message' => "Only {$available} tickets available for this category",
                 'available_quantity' => $available
             ], 422);
        }

        // Create or get session ID for holding tickets
        $sessionId = $request->session_id ?? Str::uuid()->toString();

        // Hold the tickets during checkout
        $this->ticketHoldService->hold($category->id, $quantity, $sessionId);

        // Check event capacity if enforced
        $event = \App\Models\Event::findOrFail($request->event_id);

        // Check if event has ended
        if ($event->isCompleted()) {
            return response()->json([
                'message' => 'This event has ended. Tickets are no longer available.'
            ], 422);
        }

        if ($event->enforce_capacity && $event->max_capacity) {
            $quantity = $request->quantity ?? 1; // Support multiple tickets per order

            if ($event->isAtCapacity()) {
                return response()->json([
                    'message' => 'Event is at full capacity',
                    'max_capacity' => $event->max_capacity,
                    'tickets_sold' => $event->totalTicketsSold()
                ], 422);
            }

            if (!$event->canPurchaseQuantity($quantity)) {
                $remaining = $event->remainingCapacity();
                return response()->json([
                    'message' => "Only {$remaining} tickets remaining for this event",
                    'remaining_capacity' => $remaining
                ], 422);
            }
        }

        // Calculate total amount
        $totalAmount = $category->price * $quantity;

        // Create Order
        $order = Order::create([
            'order_number' => 'ORD-' . strtoupper(Str::random(10)),
            'user_id' => $user->id,
            'event_id' => $request->event_id,
            'ticket_category_id' => $category->id,
            'amount' => $category->price,
            'quantity' => $quantity,
            'total_amount' => $totalAmount,
            'payment_status' => 'pending',
            'phone_number' => $request->phone_number,
        ]);

        // Initiate Payment
        try {
            $response = $this->daraja->initiateStkPush($order, $request->phone_number);
            
            // Save CheckoutRequestID if needed for later verification/query status
            // $order->update(['checkout_request_id' => $response['CheckoutRequestID']]); 
            // Add column if needed or store in payment_reference
            $order->update(['payment_reference' => $response['CheckoutRequestID'] ?? null]);

            return response()->json([
                'message' => 'Order created and payment initiated',
                'order' => $order,
                'session_id' => $sessionId,
                'hold_expires_at' => $this->ticketHoldService->getHoldExpiration($sessionId),
                'mpesa_response' => $response
            ], 201);

        } catch (\Exception $e) {
            // Log error
            \Log::error("Payment Initiation Failed: " . $e->getMessage());
            
            // We might want to keep the order as failed or delete it.
            // Keeping it allows retry.
            return response()->json([
                'message' => 'Order created but payment initiation failed. Please try paying again.',
                'order' => $order,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * List user orders
     */
    public function myOrders(Request $request)
    {
        $orders = Order::with(['event', 'tickets'])
            ->where('user_id', $request->user()->id)
            ->latest()
            ->paginate(10);

        return response()->json($orders);
    }

    /**
     * Get order status for polling
     */
    public function status($id)
    {
        $order = Order::with(['user', 'event', 'tickets'])->findOrFail($id);

        $response = [
            'status' => $order->payment_status,
            'order_number' => $order->order_number,
            'amount' => $order->amount,
            'phone_number' => $order->phone_number,
        ];

        if ($order->payment_status === 'paid') {
            $tickets = $order->tickets;
            if ($tickets->isNotEmpty()) {
                $response['ticket'] = [
                    'id' => $tickets->first()->id,
                    'qr_code' => $tickets->first()->qr_code,
                    'uuid' => $tickets->first()->uuid,
                    'ticket_count' => $tickets->count(),
                    'download_url' => route('tickets.download', ['filename' => 'order-' . $order->order_number . '.pdf']),
                    'tickets' => $tickets->map(function($ticket) {
                        return [
                            'id' => $ticket->id,
                            'uuid' => $ticket->uuid,
                            'qr_code' => $ticket->qr_code,
                            'download_url' => route('tickets.download', ['filename' => 'ticket-' . $ticket->uuid . '.pdf'])
                        ];
                    })
                ];
            }
        }

        return response()->json($response);
    }

    /**
     * List all orders (Admin)
     */
    public function index(Request $request)
    {
        $orders = Order::with(['user', 'event'])
            ->latest()
            ->paginate(20);

        return response()->json($orders);
    }

    /**
     * Simulate successful payment (for testing)
     */
    public function simulateSuccess($id)
    {
        $order = Order::with(['event', 'ticketCategory'])->findOrFail($id);

        if ($order->payment_status === 'paid') {
            return response()->json([
                'message' => 'Order already paid',
                'order' => $order
            ], 400);
        }

        try {
            // Mark order as paid
            $order->update([
                'payment_status' => 'paid',
            ]);

            \Log::info("Test payment success simulated for order #{$order->id}");

            // Generate tickets using TicketService
            $ticketService = app(\App\Services\TicketService::class);
            $tickets = $ticketService->generateTickets($order);

            \Log::info("Generated {$tickets->count()} ticket(s) for order #{$order->id}");

            // Load the first ticket for response
            $ticket = $tickets->first();

            return response()->json([
                'message' => 'Payment successful! Tickets generated.',
                'status' => 'paid',
                'order_number' => $order->order_number,
                'amount' => $order->total_amount ?? $order->amount,
                'phone_number' => $order->phone_number,
                'ticket_count' => $tickets->count(),
                'ticket' => [
                    'id' => $ticket->id,
                    'qr_code' => $ticket->qr_code,
                    'uuid' => $ticket->uuid,
                    'download_url' => route('tickets.download', ['filename' => 'order-' . $order->order_number . '.pdf']),
                    'tickets' => $tickets->map(function($t) {
                        return [
                            'id' => $t->id,
                            'uuid' => $t->uuid,
                            'qr_code' => $t->qr_code,
                            'download_url' => route('tickets.download', ['filename' => 'ticket-' . $t->uuid . '.pdf'])
                        ];
                    })
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error("Simulate success failed for order #{$order->id}: " . $e->getMessage());
            return response()->json([
                'message' => 'Failed to process payment simulation',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Simulate failed payment (for testing)
     */
    public function simulateFailure($id)
    {
        $order = Order::findOrFail($id);

        if ($order->payment_status === 'paid') {
            return response()->json([
                'message' => 'Cannot fail an already paid order',
                'order' => $order
            ], 400);
        }

        try {
            // Mark order as failed
            $order->update([
                'payment_status' => 'failed',
            ]);

            \Log::info("Test payment failure simulated for order #{$order->id}");

            return response()->json([
                'message' => 'Payment failed',
                'status' => 'failed',
                'order_number' => $order->order_number,
                'amount' => $order->total_amount ?? $order->amount,
                'phone_number' => $order->phone_number,
                'reason' => 'Simulated payment failure (test)'
            ]);

        } catch (\Exception $e) {
            \Log::error("Simulate failure failed for order #{$order->id}: " . $e->getMessage());
            return response()->json([
                'message' => 'Failed to record payment failure',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Find or create a guest user
     */
    private function findOrCreateGuestUser(Request $request): \App\Models\User
    {
        return \App\Models\User::firstOrCreate(
            ['email' => $request->email],
            [
                'name' => $request->full_name ?? 'Guest',
                'phone' => $request->phone_number,
                'is_guest' => true,
                'password' => bcrypt(\Illuminate\Support\Str::random(32)),
            ]
        );
    }
}
