<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Jobs\ProcessStkCallback;
use App\Jobs\ProcessPesapalCallback;
use App\Jobs\ProcessPesapalIPN;
use App\Models\Order;
use App\Models\PesapalTransaction;
use App\Services\PesapalService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    public function __construct(
        protected PesapalService $pesapalService
    ) {
    }

    /**
     * Handle M-Pesa Callback
     */
    public function callback(Request $request)
    {
        Log::info('M-Pesa Callback Received', $request->all());

        $content = $request->input('Body.stkCallback');

        if (!$content) {
             return response()->json(['result' => 'invalid_body'], 400);
        }

        // Dispatch Job
        ProcessStkCallback::dispatch($request->all());

        return response()->json(['result' => 'queued']);
    }

    /**
     * Agent initiated STK Push (for manual sales)
     */
    public function stkPush(Request $request)
    {
        return response()->json(['message' => 'Not implemented yet']);
    }

    /**
     * Initiate PesaPal Payment
     */
    public function initiatePesapal(Request $request)
    {
        try {
            $validated = $request->validate([
                'order_id' => 'required|exists:orders,id',
                'phone_number' => 'nullable|string',
                'email' => 'nullable|email',
            ]);

            $order = Order::findOrFail($validated['order_id']);

            // Check if order is already paid
            if ($order->payment_status === 'paid') {
                return response()->json([
                    'success' => false,
                    'message' => 'Order is already paid',
                ], 400);
            }

            $result = $this->pesapalService->initiatePayment(
                $order,
                $validated['phone_number'] ?? null,
                $validated['email'] ?? null
            );

            return response()->json([
                'success' => true,
                'message' => 'Payment initiated successfully',
                'data' => [
                    'order_tracking_id' => $result['order_tracking_id'],
                    'redirect_url' => $result['redirect_url'],
                ],
            ]);

        } catch (\Exception $e) {
            Log::error('PesaPal initiation failed', [
                'error' => $e->getMessage(),
                'request' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to initiate payment: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Handle PesaPal Callback (User redirect after payment)
     */
    public function pesapalCallback(Request $request)
    {
        Log::info('PesaPal Callback Received', $request->all());

        try {
            $orderTrackingId = $request->input('OrderTrackingId');

            if (!$orderTrackingId) {
                return response()->json(['message' => 'Invalid callback data'], 400);
            }

            // Dispatch job to process callback
            ProcessPesapalCallback::dispatch($request->all());

            // Get transaction
            $transaction = PesapalTransaction::where('order_tracking_id', $orderTrackingId)->first();

            if ($transaction && $transaction->order) {
                return response()->json([
                    'success' => true,
                    'message' => 'Payment processing',
                    'order_number' => $transaction->order->order_number,
                    'status' => $transaction->status,
                ]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Payment received, processing in background',
            ]);

        } catch (\Exception $e) {
            Log::error('PesaPal callback error', [
                'error' => $e->getMessage(),
                'request' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Callback processing failed',
            ], 500);
        }
    }

    /**
     * Handle PesaPal IPN (Instant Payment Notification)
     */
    public function pesapalIPN(Request $request)
    {
        Log::info('PesaPal IPN Received', $request->all());

        try {
            $orderTrackingId = $request->input('OrderTrackingId');
            $orderNotificationType = $request->input('OrderNotificationType');

            if (!$orderTrackingId) {
                return response()->json(['message' => 'Invalid IPN data'], 400);
            }

            // Dispatch job to process IPN
            ProcessPesapalIPN::dispatch($request->all());

            return response()->json([
                'success' => true,
                'message' => 'IPN received',
            ]);

        } catch (\Exception $e) {
            Log::error('PesaPal IPN error', [
                'error' => $e->getMessage(),
                'request' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'IPN processing failed',
            ], 500);
        }
    }

    /**
     * Check PesaPal Transaction Status
     */
    public function checkPesapalStatus($orderTrackingId)
    {
        try {
            $transaction = PesapalTransaction::where('order_tracking_id', $orderTrackingId)->first();

            if (!$transaction) {
                return response()->json([
                    'success' => false,
                    'message' => 'Transaction not found',
                ], 404);
            }

            // Get fresh status from PesaPal
            $statusData = $this->pesapalService->getTransactionStatus($orderTrackingId);

            return response()->json([
                'success' => true,
                'data' => [
                    'order_tracking_id' => $orderTrackingId,
                    'merchant_reference' => $transaction->merchant_reference,
                    'status' => $transaction->status,
                    'payment_method' => $statusData['payment_method'] ?? null,
                    'payment_status_description' => $statusData['payment_status_description'] ?? null,
                    'confirmation_code' => $statusData['confirmation_code'] ?? null,
                ],
            ]);

        } catch (\Exception $e) {
            Log::error('PesaPal status check error', [
                'error' => $e->getMessage(),
                'order_tracking_id' => $orderTrackingId,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to check status',
            ], 500);
        }
    }
}
