<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\SettingsService;
use Illuminate\Http\JsonResponse;

class PublicSettingsController extends Controller
{
    protected SettingsService $settings;

    public function __construct(SettingsService $settings)
    {
        $this->settings = $settings;
    }

    /**
     * Get public settings (non-sensitive configuration)
     *
     * @return JsonResponse
     */
    public function index(): JsonResponse
    {
        $publicSettings = [
            'business' => [
                'name' => $this->settings->get('business.name', config('app.name', 'MATIKO')),
                'email' => $this->settings->get('business.email', 'info@matiko.com'),
                'phone' => $this->settings->get('business.phone', '+254700000000'),
                'address' => $this->settings->get('business.address', 'Nairobi, Kenya'),
                'description' => $this->settings->get('business.description', 'Your premier event ticketing platform'),
                'logo_url' => $this->settings->get('business.logo_url'),
                'favicon_url' => $this->settings->get('business.favicon_url'),
            ],
            'seo' => [
                'title' => $this->settings->get('seo.title', 'MATIKO - Event Ticketing Platform'),
                'description' => $this->settings->get('seo.description', 'Book tickets for the best events in Kenya.'),
                'keywords' => $this->settings->get('seo.keywords', 'events, tickets, kenya'),
            ],
            'social' => [
                'facebook' => $this->settings->get('social.facebook'),
                'twitter' => $this->settings->get('social.twitter'),
                'instagram' => $this->settings->get('social.instagram'),
                'linkedin' => $this->settings->get('social.linkedin'),
            ],
            'features' => [
                'guest_checkout' => $this->settings->get('features.guest_checkout', 'true') === 'true',
                'mpesa_enabled' => $this->settings->get('features.mpesa_enabled', 'true') === 'true',
                'airtel_money_enabled' => $this->settings->get('features.airtel_money_enabled', 'true') === 'true',
            ],
        ];

        return response()->json([
            'success' => true,
            'data' => $publicSettings,
        ]);
    }

    /**
     * Get a specific setting group
     *
     * @param string $group
     * @return JsonResponse
     */
    public function getGroup(string $group): JsonResponse
    {
        // Only allow public groups
        $allowedGroups = ['business', 'seo', 'social', 'features'];

        if (!in_array($group, $allowedGroups)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid settings group',
            ], 400);
        }

        $settings = $this->settings->getGroup($group);

        // Filter out encrypted/sensitive settings
        $publicSettings = collect($settings)->filter(function ($setting) {
            return !$setting->is_encrypted;
        })->mapWithKeys(function ($setting) {
            $key = str_replace($setting->group . '.', '', $setting->key);
            return [$key => $setting->value];
        });

        return response()->json([
            'success' => true,
            'data' => $publicSettings,
        ]);
    }
}
