<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\TicketCategory;
use App\Services\DarajaService;
use App\Services\PesapalService;
use App\Services\SettingsService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class OrderController extends Controller
{
    protected $daraja;
    protected $pesapal;
    protected $settings;

    public function __construct(DarajaService $daraja, PesapalService $pesapal, SettingsService $settings)
    {
        $this->daraja = $daraja;
        $this->pesapal = $pesapal;
        $this->settings = $settings;
    }

    public function checkout(Request $request, $eventId)
    {
        // If category is specified in query params, show that category
        // Otherwise, redirect to event page to select category
        if ($request->has('category')) {
            $category = TicketCategory::with('event')->findOrFail($request->category);

            // Check if event has ended
            if ($category->event->isCompleted()) {
                return redirect()->route('events.show', $eventId)
                    ->with('error', 'This event has ended. Tickets are no longer available.');
            }

            // Get feature flags
            $pesapalEnabled = $this->settings->get('features.pesapal_enabled') === 'true';
            $mpesaEnabled = $this->settings->get('features.mpesa_enabled') === 'true';

            return view('orders.checkout', compact('category', 'pesapalEnabled', 'mpesaEnabled'));
        } else {
            // Redirect to event page to select ticket category
            return redirect()->route('events.show', $eventId)
                ->with('info', 'Please select a ticket category to continue.');
        }
    }

    public function store(Request $request)
    {
        $request->validate([
            'event_id' => 'required',
            'ticket_category_id' => 'required',
            'payment_method' => 'required|in:mpesa,pesapal',
            'phone_number' => 'required_if:payment_method,mpesa',
            'quantity' => 'nullable|integer|min:1|max:10',
        ]);

        $category = TicketCategory::with('event')->findOrFail($request->ticket_category_id);

        // Check if event has ended
        if ($category->event->isCompleted()) {
            return back()->with('error', 'This event has ended. Tickets are no longer available.');
        }

        $user = auth()->user();

        if (!$user) {
            $request->validate([
                'name' => 'required|string',
                'email' => 'nullable|email',
            ]);

            // Check if user exists
            if ($request->email && \App\Models\User::where('email', $request->email)->exists()) {
                 return back()->with('error', 'Account exists. Please login to continue.');
            }

            // Create Guest User
            $password = Str::random(12);
            $user = \App\Models\User::create([
                'name' => $request->name,
                'email' => $request->email ?? Str::slug($request->name).rand(1000,9999).'@guest.matiko.com',
                'phone' => $request->phone_number,
                'password' => \Illuminate\Support\Facades\Hash::make($password),
                'role' => 'customer',
                'status' => 'active',
            ]);

            $user->assignRole('customer');
            auth()->login($user);
        }

        $quantity = $request->input('quantity', 1);
        $totalAmount = $category->price * $quantity;
        $paymentMethod = $request->payment_method;

        $order = Order::create([
            'order_number' => 'ORD-' . strtoupper(Str::random(10)),
            'user_id' => $user->id,
            'event_id' => $request->event_id,
            'ticket_category_id' => $category->id,
            'quantity' => $quantity,
            'amount' => $category->price,
            'total_amount' => $totalAmount,
            'payment_status' => 'pending',
            'payment_method' => $paymentMethod,
            'phone_number' => $request->phone_number,
        ]);

        try {
            if ($paymentMethod === 'mpesa') {
                // M-Pesa STK Push
                $response = $this->daraja->initiateStkPush($order, $request->phone_number);
                $order->update(['payment_reference' => $response['CheckoutRequestID'] ?? null]);

                return redirect()->route('orders.success', $order->id)
                    ->with('success', 'Payment initiated! Check your phone for M-Pesa prompt.');

            } elseif ($paymentMethod === 'pesapal') {
                // PesaPal Payment
                $response = $this->pesapal->initiatePayment($order);

                if (isset($response['redirect_url'])) {
                    $order->update([
                        'payment_reference' => $response['order_tracking_id'] ?? null
                    ]);

                    // Redirect to PesaPal payment page
                    return redirect($response['redirect_url']);
                } else {
                    throw new \Exception('Failed to initialize PesaPal payment');
                }
            }

        } catch (\Exception $e) {
            return back()->with('error', 'Payment Initiation Failed: ' . $e->getMessage());
        }
    }

    public function success($id)
    {
        $order = Order::findOrFail($id);
        return view('orders.success', compact('order'));
    }

    public function myTickets()
    {
        $orders = Order::with(['event', 'ticketCategory', 'tickets'])
            ->where('user_id', auth()->id())
            ->where('payment_status', 'completed')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('orders.my-tickets', compact('orders'));
    }

    public function downloadTicket($filename)
    {
        // Validate filename to prevent directory traversal attacks
        $filename = basename($filename);

        $path = storage_path('app/public/tickets/' . $filename);

        // Check if file exists
        if (!file_exists($path)) {
            // Ensure directory exists
            $directory = dirname($path);
            if (!is_dir($directory)) {
                mkdir($directory, 0755, true);
            }

            // Generate QR code for test ticket
            $ticketCode = str_replace(['ticket-', '.pdf'], '', $filename);
            $qrContent = json_encode([
                'ticket_id' => 'test-uuid-' . $ticketCode,
                'event_id' => 1,
                'checksum' => 'test-checksum'
            ]);

            // Use PNG output (best DomPDF support)
            $options = new \chillerlan\QRCode\QROptions([
                'version'    => 5,
                'outputType' => \chillerlan\QRCode\QRCode::OUTPUT_IMAGE_PNG,
                'eccLevel'   => \chillerlan\QRCode\QRCode::ECC_L,
                'scale'      => 10,
                'imageBase64' => true,
            ]);

            // Generate QR code as PNG (best for DomPDF)
            try {
                $qrcodeData = (new \chillerlan\QRCode\QRCode($options))->render($qrContent);
                $qrcodeSvg = '<img src="' . $qrcodeData . '" alt="QR Code" style="width: 150px; height: 150px;" />';

                // Debug: Log QR code generation
                \Log::info('QR Code PNG generated for ticket: ' . $ticketCode);
                \Log::info('QR Code data length: ' . strlen($qrcodeData));
            } catch (\Exception $e) {
                \Log::error('QR Code generation failed: ' . $e->getMessage());
                $qrcodeSvg = '<div style="width: 150px; height: 150px; background: red; color: white; display: flex; align-items: center; justify-content: center;">QR Error</div>'; // Fallback
            }

            // Create a test PDF with QR code using a basic HTML template
            $debugInfo = "QR Code Type: PNG Base64 | Size: " . strlen($qrcodeData) . " chars";
            $html = '
            <!DOCTYPE html>
            <html>
            <head>
                <meta charset="utf-8">
                <title>Test Ticket</title>
                <style>
                    body { font-family: Arial, sans-serif; text-align: center; padding: 20px; }
                    .ticket { border: 2px solid #000; padding: 20px; max-width: 500px; margin: 0 auto; }
                    h1 { color: #333; margin-bottom: 20px; }
                    p { margin: 8px 0; font-size: 12px; }
                    .qr-code { margin: 20px 0; padding: 10px; }
                    .qr-code img { width: 150px; height: 150px; display: block; margin: 0 auto; }
                    .debug { background: #ffebee; padding: 10px; margin: 10px 0; font-size: 10px; color: #c62828; }
                    .footer { margin-top: 20px; font-size: 10px; color: #666; }
                </style>
            </head>
            <body>
                <div class="ticket">
                    <h1>Test Event Ticket</h1>
                    <p><strong>Ticket Code:</strong> ' . $ticketCode . '</p>
                    <p><strong>Event:</strong> Test Event</p>
                    <p><strong>Date:</strong> January 16, 2026</p>
                    <p><strong>Venue:</strong> Test Venue</p>

                    <div class="debug">
                        <strong>DEBUG INFO:</strong><br>
                        ' . $debugInfo . '<br>
                        QR Content: ' . htmlspecialchars(substr($qrContent, 0, 50)) . '...<br>
                        Generated at: ' . date('Y-m-d H:i:s') . '
                    </div>

                    <div class="qr-code">
                        <p><strong>QR CODE BELOW:</strong></p>
                        ' . $qrcodeSvg . '
                        <p><small>Scan at gate for entry</small></p>
                    </div>

                    <div class="footer">
                        <p>Ordered by: Test User | Order #TEST-ORDER</p>
                        <p>Powered by MATIKO Ticketing</p>
                    </div>
                </div>
            </body>
            </html>';

            // Use DomPDF to create the PDF with options for SVG
            try {
                \Log::info('Creating PDF for ticket: ' . $filename);
                \Log::info('HTML length: ' . strlen($html));

                $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadHTML($html);
                $pdf->getDomPDF()->set_option('isRemoteEnabled', true);
                $pdf->getDomPDF()->set_option('isHtml5ParserEnabled', true);
                $pdf->getDomPDF()->set_option('isPhpEnabled', true);

                // Set paper size and orientation
                $pdf->setPaper('A4', 'portrait');

                $pdf->save($path);

                \Log::info('PDF saved successfully to: ' . $path);
                \Log::info('File exists: ' . (file_exists($path) ? 'YES' : 'NO'));
                \Log::info('File size: ' . (file_exists($path) ? filesize($path) : 'N/A'));

            } catch (\Exception $e) {
                \Log::error('PDF generation failed: ' . $e->getMessage());
                // Create a simple fallback PDF
                $fallbackHtml = '<h1>PDF Generation Error</h1><p>' . $e->getMessage() . '</p>';
                $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadHTML($fallbackHtml);
                $pdf->save($path);
            }
        }

        // Verify file exists before serving
        if (!file_exists($path)) {
            abort(404, 'Ticket not found');
        }

        return response()->file($path, [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => 'inline; filename="' . $filename . '"'
        ]);
    }
}
