<?php

namespace App\Jobs;

use App\Models\PesapalTransaction;
use App\Services\PesapalService;
use App\Services\TicketService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessPesapalCallback implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $payload;

    /**
     * Create a new job instance.
     */
    public function __construct(array $payload)
    {
        $this->payload = $payload;
    }

    /**
     * Execute the job.
     */
    public function handle(PesapalService $pesapalService, TicketService $ticketService): void
    {
        try {
            $orderTrackingId = $this->payload['OrderTrackingId'] ?? null;

            if (!$orderTrackingId) {
                Log::warning('PesaPal callback received without OrderTrackingId', $this->payload);
                return;
            }

            Log::info("Processing PesaPal callback for tracking ID: {$orderTrackingId}");

            // Find the transaction
            $transaction = PesapalTransaction::where('order_tracking_id', $orderTrackingId)->first();

            if (!$transaction) {
                Log::warning("PesaPal transaction not found for tracking ID: {$orderTrackingId}");
                return;
            }

            // Update transaction from callback
            $updatedTransaction = $pesapalService->updateTransactionFromCallback($this->payload);

            if (!$updatedTransaction) {
                Log::error("Failed to update transaction from callback", [
                    'order_tracking_id' => $orderTrackingId,
                ]);
                return;
            }

            // If payment was completed, generate tickets
            if ($updatedTransaction->isCompleted() && $updatedTransaction->order) {
                Log::info("Payment completed for Order {$updatedTransaction->order->order_number}, generating tickets");

                try {
                    $ticketService->generateTickets($updatedTransaction->order);
                    Log::info("Tickets generated successfully for Order {$updatedTransaction->order->order_number}");
                } catch (\Exception $e) {
                    Log::error("Failed to generate tickets for Order {$updatedTransaction->order->id}: " . $e->getMessage());
                }
            } elseif ($updatedTransaction->isFailed()) {
                Log::warning("Payment failed for Order {$updatedTransaction->order->order_number}: {$updatedTransaction->payment_status_description}");
            }

        } catch (\Exception $e) {
            Log::error('PesaPal callback processing error', [
                'payload' => $this->payload,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            throw $e;
        }
    }
}
