<?php

namespace App\Jobs;

use App\Models\PesapalTransaction;
use App\Services\PesapalService;
use App\Services\TicketService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessPesapalIPN implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $payload;

    /**
     * Create a new job instance.
     */
    public function __construct(array $payload)
    {
        $this->payload = $payload;
    }

    /**
     * Execute the job.
     */
    public function handle(PesapalService $pesapalService, TicketService $ticketService): void
    {
        try {
            $orderTrackingId = $this->payload['OrderTrackingId'] ?? null;
            $orderNotificationType = $this->payload['OrderNotificationType'] ?? null;

            if (!$orderTrackingId) {
                Log::warning('PesaPal IPN received without OrderTrackingId', $this->payload);
                return;
            }

            Log::info("Processing PesaPal IPN for tracking ID: {$orderTrackingId}, Type: {$orderNotificationType}");

            // Find the transaction
            $transaction = PesapalTransaction::where('order_tracking_id', $orderTrackingId)->first();

            if (!$transaction) {
                Log::warning("PesaPal transaction not found for tracking ID: {$orderTrackingId}");
                return;
            }

            // Store IPN data
            $transaction->update([
                'ipn_data' => array_merge($transaction->ipn_data ?? [], [
                    [
                        'timestamp' => now()->toIso8601String(),
                        'type' => $orderNotificationType,
                        'data' => $this->payload,
                    ]
                ]),
            ]);

            // Get fresh transaction status from PesaPal
            $statusData = $pesapalService->getTransactionStatus($orderTrackingId);

            // Update transaction status
            $status = $pesapalService->updateTransactionFromCallback([
                'OrderTrackingId' => $orderTrackingId,
            ]);

            if (!$status) {
                Log::error("Failed to update transaction from IPN", [
                    'order_tracking_id' => $orderTrackingId,
                ]);
                return;
            }

            // If payment was completed and tickets haven't been generated yet, generate them
            if ($status->isCompleted() && $status->order) {
                $order = $status->order;

                // Check if tickets already exist
                if ($order->tickets()->count() === 0) {
                    Log::info("Payment completed via IPN for Order {$order->order_number}, generating tickets");

                    try {
                        $ticketService->generateTickets($order);
                        Log::info("Tickets generated successfully for Order {$order->order_number}");
                    } catch (\Exception $e) {
                        Log::error("Failed to generate tickets for Order {$order->id}: " . $e->getMessage());
                    }
                } else {
                    Log::info("Tickets already exist for Order {$order->order_number}, skipping generation");
                }
            }

        } catch (\Exception $e) {
            Log::error('PesaPal IPN processing error', [
                'payload' => $this->payload,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            throw $e;
        }
    }
}
