<?php

namespace App\Jobs;

use App\Models\Order;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class SendTicketEmailJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $orderId;

    /**
     * Create a new job instance.
     */
    public function __construct($orderId)
    {
        $this->orderId = $orderId;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $order = Order::with(['user', 'event', 'tickets'])->find($this->orderId);

        if (!$order) {
            Log::error("Order not found for email job: {$this->orderId}");
            return;
        }

        $user = $order->user;
        $event = $order->event;
        $ticketCount = $order->tickets->count();

        // Skip if user has no email
        if (!$user->email) {
            Log::info("No email address for user {$user->id}, skipping email");
            return;
        }

        // Prepare attachment - now using order-based PDF
        $pdfPath = storage_path('app/public/tickets/order-' . $order->order_number . '.pdf');

        if (!file_exists($pdfPath)) {
            Log::error("PDF not found for order: {$order->order_number}");
            return;
        }

        try {
            // Prepare email body with ticket count
            $ticketText = $ticketCount > 1 ? "{$ticketCount} tickets" : "ticket";
            $haveHas = $ticketCount > 1 ? "have" : "has";

            $emailBody = "Dear {$user->name},\n\n";
            $emailBody .= "Thank you for your purchase! Your {$ticketText} for {$event->name} {$haveHas} been successfully confirmed.\n\n";
            $emailBody .= "Event Details:\n";
            $emailBody .= "- Date: " . \Carbon\Carbon::parse($event->event_date)->format('l, F j, Y') . "\n";
            $emailBody .= "- Time: {$event->start_time} - {$event->end_time}\n";
            $emailBody .= "- Venue: {$event->venue}\n\n";

            if ($ticketCount > 1) {
                $emailBody .= "Your PDF contains {$ticketCount} tickets, one per page. Each ticket has a unique QR code.\n";
                $emailBody .= "Please ensure all attendees have their respective QR codes for entry.\n\n";
            } else {
                $emailBody .= "Please find your ticket attached to this email.\n\n";
            }

            $emailBody .= "Present the QR code(s) at the entrance for scanning.\n\n";
            $emailBody .= "See you at the event!\n\n";
            $emailBody .= "Best regards,\nMATIKO Team";

            // Send email using Laravel Mail
            Mail::raw(
                $emailBody,
                function ($message) use ($user, $event, $pdfPath, $order) {
                    $message->to($user->email, $user->name)
                        ->subject("Your Tickets for {$event->name}")
                        ->attach($pdfPath, [
                            'as' => 'tickets-' . $order->order_number . '.pdf',
                            'mime' => 'application/pdf',
                        ]);
                }
            );

            Log::info("Email sent successfully for order {$order->id} to {$user->email}");
        } catch (\Exception $e) {
            Log::error("Failed to send email for order {$order->id}: " . $e->getMessage());
            // Don't throw - allow the job to complete so ticket generation succeeds
        }
    }
}
