<?php

namespace App\Jobs;

use App\Models\Order;
use App\Services\AfricasTalkingService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class SendTicketSMSJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $orderId;

    /**
     * Create a new job instance.
     */
    public function __construct($orderId)
    {
        $this->orderId = $orderId;
    }

    /**
     * Execute the job.
     */
    public function handle(AfricasTalkingService $atService): void
    {
        $order = Order::with(['user', 'event', 'tickets'])->find($this->orderId);

        if (!$order) {
            Log::error("Order not found for SMS job: {$this->orderId}");
            return;
        }

        $user = $order->user;
        $event = $order->event;
        $ticketCount = $order->tickets->count();

        // Prepare SMS content
        $downloadUrl = url('storage/tickets/order-' . $order->order_number . '.pdf');
        $ticketText = $ticketCount > 1 ? "{$ticketCount} tickets" : "ticket";

        $message = "MATIKO: Your {$ticketText} for {$event->name}. Download: {$downloadUrl}";

        // Send SMS
        $result = $atService->sendSms($user->phone, $message);

        if (isset($result['status']) && $result['status'] === 'error') {
            Log::error("Failed to send SMS for order {$order->id}: " . $result['message']);
            throw new \Exception("SMS sending failed: " . $result['message']);
        }

        Log::info("SMS sent successfully for order {$order->id} to {$user->phone}");
    }
}
