<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;

class Event extends Model
{
    /** @use HasFactory<\Database\Factories\EventFactory> */
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'venue',
        'event_date',
        'start_time',
        'end_time',
        'max_capacity',
        'enforce_capacity',
        'status',
        'created_by',
        'image_path',
        'featured',
        'category',
    ];

    protected $casts = [
        'event_date' => 'date',
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'featured' => 'boolean',
        'enforce_capacity' => 'boolean',
        'max_capacity' => 'integer',
    ];

    protected $appends = [
        'image_url',
        'min_price',
    ];

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function categories(): HasMany
    {
        return $this->hasMany(TicketCategory::class);
    }

    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    public function tickets(): HasManyThrough
    {
        return $this->hasManyThrough(Ticket::class, TicketCategory::class);
    }

    /**
     * Get the event's image URL
     */
    protected function imageUrl(): Attribute
    {
        return Attribute::make(
            get: function () {
                if (!$this->image_path) {
                    return null;
                }

                // If it's already a full URL (e.g., from external CDN), return as-is
                if (str_starts_with($this->image_path, 'http')) {
                    return $this->image_path;
                }

                // Otherwise, generate storage URL
                return asset('storage/' . $this->image_path);
            }
        );
    }

    /**
     * Get the minimum ticket price for this event
     */
    protected function minPrice(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->categories()->min('price')
        );
    }

    /**
     * Get total tickets sold across all categories
     */
    public function totalTicketsSold(): int
    {
        return $this->tickets()->count();
    }

    /**
     * Get remaining capacity for the event
     */
    public function remainingCapacity(): ?int
    {
        if (!$this->max_capacity) {
            return null;
        }

        return $this->max_capacity - $this->totalTicketsSold();
    }

    /**
     * Check if event is at capacity
     */
    public function isAtCapacity(): bool
    {
        if (!$this->enforce_capacity || !$this->max_capacity) {
            return false;
        }

        return $this->totalTicketsSold() >= $this->max_capacity;
    }

    /**
     * Check if a specific quantity can be purchased
     */
    public function canPurchaseQuantity(int $quantity): bool
    {
        if (!$this->enforce_capacity || !$this->max_capacity) {
            return true;
        }

        return ($this->totalTicketsSold() + $quantity) <= $this->max_capacity;
    }

    /**
     * Check if event is completed (past)
     */
    public function isCompleted(): bool
    {
        return $this->event_date->isPast() || $this->status === 'completed';
    }

    /**
     * Check if event is ongoing (happening today)
     */
    public function isOngoing(): bool
    {
        return $this->event_date->isToday() || $this->status === 'ongoing';
    }

    /**
     * Check if event is upcoming (future)
     */
    public function isUpcoming(): bool
    {
        return $this->event_date->isFuture() || $this->status === 'upcoming';
    }

    /**
     * Get the display status of the event
     */
    public function getDisplayStatus(): string
    {
        if ($this->isCompleted()) {
            return 'completed';
        }

        if ($this->isOngoing()) {
            return 'ongoing';
        }

        return 'upcoming';
    }
}
