<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\Attribute;

class MenuItem extends Model
{
    protected $fillable = [
        'menu_id',
        'parent_id',
        'label',
        'url',
        'url_type',
        'page_id',
        'target',
        'order',
        'attributes'
    ];

    protected $casts = [
        'attributes' => 'array',
    ];

    /**
     * Get the menu this item belongs to.
     */
    public function menu()
    {
        return $this->belongsTo(Menu::class);
    }

    /**
     * Get the parent menu item.
     */
    public function parent()
    {
        return $this->belongsTo(MenuItem::class, 'parent_id');
    }

    /**
     * Get child menu items.
     */
    public function children()
    {
        return $this->hasMany(MenuItem::class, 'parent_id')->orderBy('order');
    }

    /**
     * Get the linked CMS page.
     */
    public function page()
    {
        return $this->belongsTo(Page::class);
    }

    /**
     * Get the resolved URL based on url_type.
     */
    protected function resolvedUrl(): Attribute
    {
        return Attribute::make(
            get: function () {
                return match($this->url_type) {
                    'page' => $this->page ? url("/page/{$this->page->slug}") : '#',
                    'route' => $this->resolveRoute(),
                    default => $this->url ?? '#',
                };
            }
        );
    }

    /**
     * Safely resolve a named route.
     */
    private function resolveRoute(): string
    {
        if (!$this->url) {
            return '#';
        }

        try {
            return route($this->url);
        } catch (\Exception $e) {
            \Log::warning("Invalid route name in menu item: {$this->url}");
            return '#';
        }
    }
}
