<?php

namespace App\Services;

use App\Models\Order;
use App\Models\PesapalTransaction;
use Illuminate\Support\Facades\Log;
use NjoguAmos\Pesapal\Pesapal;
use NjoguAmos\Pesapal\DTOs\PesapalOrderData;
use NjoguAmos\Pesapal\DTOs\PesapalAddressData;
use NjoguAmos\Pesapal\Enums\IpnType;

class PesapalService
{
    public function __construct(protected SettingsService $settings)
    {
    }

    /**
     * Initiate PesaPal payment
     *
     * @param Order $order
     * @param string|null $phoneNumber
     * @param string|null $email
     * @return array
     */
    public function initiatePayment(Order $order, $phoneNumber = null, $email = null)
    {
        try {
            // Use order phone number if not provided
            $phoneNumber = $phoneNumber ?? $order->phone_number;
            $email = $email ?? $order->user?->email ?? 'customer@example.com';

            // Format phone number
            $phone = $this->formatPhoneNumber($phoneNumber);

            // Get the event details
            $event = $order->event;
            $ticketCategory = $order->ticketCategory;

            // Description for the payment
            $description = sprintf(
                '%s - %s (x%d)',
                $event->name ?? 'Event Ticket',
                $ticketCategory->name ?? 'Ticket',
                $order->quantity
            );

            // Use total_amount if available, otherwise use amount
            $paymentAmount = $order->total_amount ?? $order->amount;

            // Create order data for PesaPal
            $orderData = PesapalOrderData::from([
                'id' => $order->order_number,
                'currency' => 'KES',
                'amount' => (float) $paymentAmount,
                'description' => $description,
                'callback_url' => route('api.pesapal.callback'),
                'notification_id' => $this->getOrCreateIpnId(),
            ]);

            // Create billing address
            $billingAddress = PesapalAddressData::from([
                'phone_number' => $phone,
                'email_address' => $email,
                'first_name' => $order->user?->name ?? 'Customer',
                'last_name' => '',
            ]);

            // Submit order to PesaPal
            $result = Pesapal::createOrder($orderData, $billingAddress);

            // Check if result is an array (success) or Response object (error)
            if (is_array($result)) {
                // Create transaction record
                $transaction = PesapalTransaction::create([
                    'order_tracking_id' => $result['order_tracking_id'],
                    'merchant_reference' => $order->order_number,
                    'order_id' => $order->id,
                    'amount' => $paymentAmount,
                    'currency' => 'KES',
                    'status' => PesapalTransaction::STATUS_PENDING,
                    'description' => $description,
                    'redirect_url' => Pesapal::getRedirectUrl($result['order_tracking_id']),
                ]);

                Log::info('PesaPal payment initiated', [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number,
                    'tracking_id' => $result['order_tracking_id'],
                ]);

                return [
                    'success' => true,
                    'order_tracking_id' => $result['order_tracking_id'],
                    'redirect_url' => $transaction->redirect_url,
                    'transaction' => $transaction,
                ];
            }

            // Handle error response
            Log::error('PesaPal order creation failed', [
                'order_id' => $order->id,
                'response' => $result->body(),
            ]);

            throw new \Exception('Failed to create PesaPal order: ' . ($result->json()['error']['message'] ?? 'Unknown error'));

        } catch (\Exception $e) {
            Log::error('PesaPal payment initiation error', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    /**
     * Get transaction status from PesaPal
     *
     * @param string $orderTrackingId
     * @return array
     */
    public function getTransactionStatus($orderTrackingId)
    {
        try {
            $result = Pesapal::getTransactionStatus($orderTrackingId);

            if (is_array($result)) {
                Log::info('PesaPal transaction status retrieved', [
                    'order_tracking_id' => $orderTrackingId,
                    'status' => $result['payment_status_description'] ?? 'unknown',
                ]);

                return $result;
            }

            Log::error('PesaPal status check failed', [
                'order_tracking_id' => $orderTrackingId,
                'response' => $result->body(),
            ]);

            throw new \Exception('Failed to get transaction status');

        } catch (\Exception $e) {
            Log::error('PesaPal status check error', [
                'order_tracking_id' => $orderTrackingId,
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    /**
     * Update transaction from callback/IPN data
     *
     * @param array $data
     * @return PesapalTransaction|null
     */
    public function updateTransactionFromCallback(array $data)
    {
        try {
            $orderTrackingId = $data['OrderTrackingId'] ?? null;

            if (!$orderTrackingId) {
                Log::warning('PesaPal callback received without OrderTrackingId', $data);
                return null;
            }

            $transaction = PesapalTransaction::where('order_tracking_id', $orderTrackingId)->first();

            if (!$transaction) {
                Log::warning('PesaPal transaction not found', [
                    'order_tracking_id' => $orderTrackingId,
                ]);
                return null;
            }

            // Get fresh status from PesaPal
            $statusData = $this->getTransactionStatus($orderTrackingId);

            // Update transaction
            $status = $this->mapPesapalStatus($statusData['payment_status_description'] ?? '');

            $transaction->update([
                'status' => $status,
                'payment_method' => $statusData['payment_method'] ?? null,
                'confirmation_code' => $statusData['confirmation_code'] ?? null,
                'payment_status_description' => $statusData['payment_status_description'] ?? null,
                'callback_data' => $statusData,
                'completed_at' => $status === PesapalTransaction::STATUS_COMPLETED ? now() : null,
            ]);

            // Update order if payment completed
            if ($status === PesapalTransaction::STATUS_COMPLETED) {
                $transaction->order->update([
                    'payment_status' => 'paid',
                    'payment_reference' => $statusData['confirmation_code'] ?? $orderTrackingId,
                ]);
            }

            return $transaction;

        } catch (\Exception $e) {
            Log::error('PesaPal callback processing error', [
                'data' => $data,
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    /**
     * Get or create IPN ID
     *
     * @return string
     */
    protected function getOrCreateIpnId()
    {
        try {
            // Check if we have a registered IPN
            $ipn = \NjoguAmos\Pesapal\Models\PesapalIpn::where('status', 1)->first();

            if ($ipn) {
                return $ipn->ipn_id;
            }

            // Register new IPN
            $result = Pesapal::createIpn(
                url: route('api.pesapal.ipn'),
                ipnType: IpnType::GET
            );

            if ($result instanceof \NjoguAmos\Pesapal\Models\PesapalIpn) {
                return $result->ipn_id;
            }

            Log::warning('Failed to register IPN, using empty string');
            return '';

        } catch (\Exception $e) {
            Log::error('IPN registration error', ['error' => $e->getMessage()]);
            return '';
        }
    }

    /**
     * Map PesaPal status to our status constants
     *
     * @param string $pesapalStatus
     * @return string
     */
    protected function mapPesapalStatus($pesapalStatus)
    {
        $status = strtoupper($pesapalStatus);

        return match (true) {
            str_contains($status, 'COMPLETED') => PesapalTransaction::STATUS_COMPLETED,
            str_contains($status, 'FAILED') => PesapalTransaction::STATUS_FAILED,
            str_contains($status, 'CANCELLED') || str_contains($status, 'INVALID') => PesapalTransaction::STATUS_CANCELLED,
            default => PesapalTransaction::STATUS_PENDING,
        };
    }

    /**
     * Format phone number to international format
     *
     * @param string $phone
     * @return string
     */
    protected function formatPhoneNumber($phone)
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);

        if (str_starts_with($phone, '0')) {
            return '254' . substr($phone, 1);
        }

        if (str_starts_with($phone, '7') || str_starts_with($phone, '1')) {
            return '254' . $phone;
        }

        return $phone;
    }
}
