<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Facades\Cache;

class SettingsService
{
    /**
     * Get a setting value by key.
     * Falls back to config if not found in database.
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function get(string $key, $default = null)
    {
        return Cache::remember("setting.{$key}", 3600, function () use ($key, $default) {
            $setting = Setting::where('key', $key)->first();

            if ($setting) {
                return $setting->value;
            }

            // Fallback to config
            $configKey = str_replace('.', '.', $key);
            return config("services.{$configKey}", $default);
        });
    }

    /**
     * Set a setting value.
     *
     * @param string $key
     * @param mixed $value
     * @param string|null $group
     * @param bool $encrypted
     * @return void
     */
    public function set(string $key, $value, string $group = null, bool $encrypted = false)
    {
        Setting::updateOrCreate(
            ['key' => $key],
            [
                'value' => $value,
                'group' => $group,
                'is_encrypted' => $encrypted,
            ]
        );

        Cache::forget("setting.{$key}");
    }

    /**
     * Get all settings for a specific group.
     *
     * @param string $group
     * @return array
     */
    public function getGroup(string $group): array
    {
        return Setting::where('group', $group)->pluck('value', 'key')->toArray();
    }

    /**
     * Flush all settings cache.
     *
     * @return void
     */
    public function flush()
    {
        $settings = Setting::all();
        foreach ($settings as $setting) {
            Cache::forget("setting.{$setting->key}");
        }
    }

    /**
     * Check if a setting exists.
     *
     * @param string $key
     * @return bool
     */
    public function has(string $key): bool
    {
        return Setting::where('key', $key)->exists();
    }

    /**
     * Delete a setting.
     *
     * @param string $key
     * @return void
     */
    public function forget(string $key)
    {
        Setting::where('key', $key)->delete();
        Cache::forget("setting.{$key}");
    }
}
