<?php

namespace App\Services;

use App\Models\Ticket;
use App\Models\TicketCategory;
use App\Models\TicketHold;
use Carbon\Carbon;

class TicketHoldService
{
    /**
     * Hold time in minutes
     */
    protected int $holdDurationMinutes = 10;

    /**
     * Create a hold for tickets
     */
    public function hold(int $categoryId, int $quantity, string $sessionId): TicketHold
    {
        // Release any existing holds for this session
        $this->release($sessionId);

        return TicketHold::create([
            'ticket_category_id' => $categoryId,
            'session_id' => $sessionId,
            'quantity' => $quantity,
            'expires_at' => now()->addMinutes($this->holdDurationMinutes),
        ]);
    }

    /**
     * Release holds for a specific session
     */
    public function release(string $sessionId): void
    {
        TicketHold::where('session_id', $sessionId)->delete();
    }

    /**
     * Get available quantity for a category (accounting for holds)
     */
    public function getAvailableQuantity(int $categoryId): int
    {
        $category = TicketCategory::findOrFail($categoryId);

        // Count sold tickets
        $sold = Ticket::where('ticket_category_id', $categoryId)->count();

        // Count active holds
        $held = TicketHold::where('ticket_category_id', $categoryId)
            ->active()
            ->sum('quantity');

        $available = $category->quantity - $sold - $held;

        return max(0, $available);
    }

    /**
     * Check if a specific quantity is available
     */
    public function isQuantityAvailable(int $categoryId, int $quantity): bool
    {
        return $this->getAvailableQuantity($categoryId) >= $quantity;
    }

    /**
     * Clean up expired holds (for scheduled task)
     */
    public function cleanExpiredHolds(): int
    {
        return TicketHold::where('expires_at', '<', now())->delete();
    }

    /**
     * Get hold expiration time for a session
     */
    public function getHoldExpiration(string $sessionId): ?Carbon
    {
        $hold = TicketHold::where('session_id', $sessionId)
            ->active()
            ->first();

        return $hold?->expires_at;
    }

    /**
     * Extend hold duration for a session
     */
    public function extendHold(string $sessionId, int $additionalMinutes = null): bool
    {
        $additionalMinutes = $additionalMinutes ?? $this->holdDurationMinutes;

        return TicketHold::where('session_id', $sessionId)
            ->active()
            ->update([
                'expires_at' => now()->addMinutes($additionalMinutes)
            ]) > 0;
    }
}
