<?php

namespace App\Services;

use App\Jobs\SendTicketEmailJob;
use App\Jobs\SendTicketSMSJob;
use App\Models\Order;
use App\Models\Ticket;
use App\Notifications\TicketPurchased;
use Barryvdh\DomPDF\Facade\Pdf;
use chillerlan\QRCode\QRCode;
use chillerlan\QRCode\QROptions;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class TicketService
{
    /**
     * Generate tickets for a paid order
     */
    public function generateTickets(Order $order)
    {
        // 1. Create Multiple Ticket Records based on order quantity
        $quantity = $order->quantity ?? 1;
        $tickets = collect();

        Log::info("Generating {$quantity} tickets for order #{$order->id}");

        for ($i = 0; $i < $quantity; $i++) {
            $ticket = Ticket::create([
                'order_id' => $order->id,
                'ticket_category_id' => $order->ticket_category_id ?? \App\Models\TicketCategory::where('event_id', $order->event_id)->first()->id,
                'qr_code' => Str::upper(Str::random(10)), // Placeholder, will generate unique hash
                'uuid' => Str::uuid(),
                'is_checked_in' => false,
            ]);

            // Generate checksum after creation
            $checksum = $this->generateChecksum($ticket);
            $ticket->update(['checksum' => $checksum]);

            // Update QR code with something unique and verifiable
            $ticket->update([
                'qr_code' => 'TKT-' . $order->id . '-' . $ticket->id . '-' . Str::random(5)
            ]);

            // Reload ticket with category relationship
            $ticket->load('category');

            $tickets->push($ticket);

            Log::info("Created ticket #{$ticket->id} with QR: {$ticket->qr_code}");
        }

        // 2. Generate QR codes for all tickets
        $ticketsWithQR = $tickets->map(function($ticket) use ($order) {
            $qrContent = json_encode([
                'ticket_id' => $ticket->uuid,
                'event_id' => $order->event_id,
                'checksum' => $ticket->checksum
            ]);

            $options = new QROptions([
                'version'    => QRCode::VERSION_AUTO,
                'outputType' => QRCode::OUTPUT_IMAGE_PNG,
                'eccLevel'   => QRCode::ECC_L,
                'scale'      => 10,
                'imageBase64' => true,
            ]);

            $qrcodeData = (new QRCode($options))->render($qrContent);

            return [
                'ticket' => $ticket,
                'qrcode' => '<img src="' . $qrcodeData . '" alt="QR Code" style="width: 200px; height: 200px;" />'
            ];
        });

        // 3. Generate SINGLE Multi-Page PDF for all tickets
        Log::info("Generating multi-page PDF for order #{$order->id} with {$ticketsWithQR->count()} tickets");

        $pdf = Pdf::loadView('pdfs.tickets-multi', [
            'tickets' => $ticketsWithQR,
            'order' => $order,
            'event' => $order->event,
        ]);

        // Configure DomPDF
        $pdf->getDomPDF()->set_option('isRemoteEnabled', true);
        $pdf->getDomPDF()->set_option('isHtml5ParserEnabled', true);
        $pdf->setPaper('A4', 'portrait');

        // Save single PDF for the entire order
        $pdfPath = 'tickets/order-' . $order->order_number . '.pdf';
        Storage::disk('public')->put($pdfPath, $pdf->output());

        Log::info("Multi-page PDF saved to: {$pdfPath}");

        // 3b. Generate individual ticket PDFs for each ticket
        foreach ($ticketsWithQR as $index => $ticketData) {
            $singlePdf = Pdf::loadView('pdfs.ticket', [
                'ticket' => $ticketData['ticket'],
                'qrcode' => $ticketData['qrcode'],
                'order' => $order,
                'event' => $order->event,
            ]);

            // Configure DomPDF for single ticket
            $singlePdf->getDomPDF()->set_option('isRemoteEnabled', true);
            $singlePdf->getDomPDF()->set_option('isHtml5ParserEnabled', true);
            $singlePdf->setPaper('A4', 'portrait');

            // Save individual ticket PDF
            $singlePdfPath = 'tickets/ticket-' . $ticketData['ticket']->uuid . '.pdf';
            Storage::disk('public')->put($singlePdfPath, $singlePdf->output());
        }

        // 4. Dispatch SINGLE notification job per order (not per ticket)
        SendTicketEmailJob::dispatch($order->id);
        SendTicketSMSJob::dispatch($order->id);

        return $tickets;
    }

    /**
     * Validate a ticket QR code
     */
    public function validateTicket(string $qrData): array
    {
        $data = json_decode($qrData, true);

        if (!$data || !isset($data['ticket_id'], $data['event_id'], $data['checksum'])) {
            return ['valid' => false, 'message' => 'Invalid QR code format'];
        }

        $ticket = Ticket::where('uuid', $data['ticket_id'])->first();

        if (!$ticket) {
            return ['valid' => false, 'message' => 'Ticket not found'];
        }

        // Check event matches
        if ($ticket->order->event_id != $data['event_id']) {
            return ['valid' => false, 'message' => 'Event mismatch'];
        }

        // Check not already checked in
        if ($ticket->is_checked_in) {
            return ['valid' => false, 'message' => 'Ticket already checked in'];
        }

        // Check payment successful
        if ($ticket->order->payment_status !== 'paid') {
            return ['valid' => false, 'message' => 'Payment not completed'];
        }

        // Verify checksum
        $expectedChecksum = $this->generateChecksum($ticket);
        if ($data['checksum'] !== $expectedChecksum) {
            return ['valid' => false, 'message' => 'Invalid checksum'];
        }

        return ['valid' => true, 'ticket' => $ticket];
    }

    /**
     * Generate checksum for ticket
     */
    private function generateChecksum(Ticket $ticket): string
    {
        $data = $ticket->uuid . $ticket->order->event_id . $ticket->order->user_id . config('app.key');
        return hash('sha256', $data);
    }

    protected function sendSms($phone, $ticket, $event)
    {
        // Use AfricasTalking or similar
        // Mocking for now or using the AT package if configured
        try {
            $message = "Hello! Your ticket for {$event->name} is ready. Download here: " . url('storage/tickets/ticket-' . $ticket->qr_code . '.pdf');
            // Core SMS sending logic would go here
            Log::info("SMS sent to {$phone}: {$message}");
        } catch (\Exception $e) {
            Log::error("SMS Send Error: " . $e->getMessage());
        }
    }
}
