<?php

namespace Database\Seeders;

use App\Models\User;
use App\Models\Event;
use App\Models\TicketCategory;
use App\Models\Order;
use App\Models\Ticket;
use App\Models\CheckIn;
use App\Models\Agent;
use App\Models\MpesaTransaction;
use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class DatabaseSeeder extends Seeder
{
    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        // Seed roles, permissions, and users
        $this->call(RolesAndPermissionsSeeder::class);

        // Seed settings
        $this->call(SettingsSeeder::class);

        // Get the users
        $superAdmin = User::where('email', 'superadmin@matiko.com')->first();
        $admin = User::where('email', 'admin@matiko.com')->first();
        $agent = User::where('email', 'agent@matiko.com')->first();
        $customer = User::where('email', 'customer@matiko.com')->first();

        // Seed CMS pages
        $this->call(PagesSeeder::class);

        // Seed menus (must be after pages)
        $this->call(DefaultMenusSeeder::class);

        // Create additional customers
        $customers = $this->seedCustomers();

        // Create agents
        $agents = $this->seedAgents([$agent]);

        // Seed Events
        $events = $this->seedEvents($superAdmin, $admin);

        // Seed Orders and Tickets
        $this->seedOrdersAndTickets($events, array_merge([$customer], $customers));

        // Seed Check-ins
        $this->seedCheckIns($agents);
    }

    protected function seedCustomers(): array
    {
        $customers = [];
        $names = [
            ['John Kamau', '+254723456789'],
            ['Mary Njeri', '+254734567890'],
            ['David Ochieng', '+254745678901'],
            ['Grace Akinyi', '+254756789012'],
            ['Peter Mwangi', '+254767890123'],
            ['Sarah Wanjiru', '+254778901234'],
            ['James Otieno', '+254789012345'],
            ['Lucy Chebet', '+254790123456'],
        ];

        foreach ($names as $index => $nameData) {
            $email = strtolower(str_replace(' ', '.', $nameData[0])) . '@example.com';
            $user = User::create([
                'name' => $nameData[0],
                'email' => $email,
                'phone' => $nameData[1],
                'password' => Hash::make('password'),
                'role' => 'customer',
                'status' => 'active',
                'email_verified_at' => now(),
            ]);
            $user->assignRole('customer');
            $customers[] = $user;
        }

        return $customers;
    }

    protected function seedAgents(array $existingAgents): array
    {
        $agents = [];

        // Create agent records for existing agent users
        foreach ($existingAgents as $agentUser) {
            $agent = Agent::create([
                'user_id' => $agentUser->id,
                'assigned_events' => [1, 2, 3],
            ]);
            $agents[] = $agent;
        }

        // Create additional agent users
        $agentNames = [
            ['Agent Mike', '+254701234567'],
            ['Agent Jane', '+254702345678'],
        ];

        foreach ($agentNames as $agentData) {
            $email = strtolower(str_replace(' ', '.', $agentData[0])) . '@matiko.com';
            $user = User::create([
                'name' => $agentData[0],
                'email' => $email,
                'phone' => $agentData[1],
                'password' => Hash::make('password'),
                'role' => 'agent',
                'status' => 'active',
                'email_verified_at' => now(),
            ]);
            $user->assignRole('agent');

            $agent = Agent::create([
                'user_id' => $user->id,
                'assigned_events' => [2, 3],
            ]);
            $agents[] = $agent;
        }

        return $agents;
    }

    protected function seedEvents(User $creator1, User $creator2): array
    {
        $events = [];

        // Event 1: Past event (completed)
        $event1 = Event::create([
            'name' => 'New Year Gala 2026',
            'description' => 'A spectacular New Year celebration with live performances, DJ sets, and fireworks.',
            'venue' => 'Safari Park Hotel, Nairobi',
            'event_date' => now()->subDays(25)->toDateString(),
            'start_time' => '20:00:00',
            'end_time' => '02:00:00',
            'status' => 'completed',
            'category' => 'Music & Concerts',
            'featured' => true,
            'max_capacity' => 800,
            'enforce_capacity' => true,
            'created_by' => $creator1->id,
        ]);

        TicketCategory::create(['event_id' => $event1->id, 'name' => 'Standard', 'price' => 2500.00, 'quantity' => 400]);
        TicketCategory::create(['event_id' => $event1->id, 'name' => 'VIP', 'price' => 6000.00, 'quantity' => 200]);
        TicketCategory::create(['event_id' => $event1->id, 'name' => 'VVIP', 'price' => 12000.00, 'quantity' => 100]);
        $events[] = $event1;

        // Event 2: Ongoing event
        $event2 = Event::create([
            'name' => 'Summer Music Festival',
            'description' => 'A family-friendly night of incredible live music, food trucks, and entertainment for all ages.',
            'venue' => 'Nairobi City Park',
            'event_date' => now()->toDateString(),
            'start_time' => '18:00:00',
            'end_time' => '23:59:59',
            'status' => 'ongoing',
            'category' => 'Music & Concerts',
            'featured' => true,
            'max_capacity' => 1000,
            'enforce_capacity' => true,
            'created_by' => $creator1->id,
        ]);

        TicketCategory::create(['event_id' => $event2->id, 'name' => 'Kids (5-12 years)', 'price' => 500.00, 'quantity' => 200]);
        TicketCategory::create(['event_id' => $event2->id, 'name' => 'Adults', 'price' => 1500.00, 'quantity' => 600]);
        TicketCategory::create(['event_id' => $event2->id, 'name' => 'VIP', 'price' => 5000.00, 'quantity' => 100]);
        TicketCategory::create(['event_id' => $event2->id, 'name' => 'Family Pass (2 Adults + 2 Kids)', 'price' => 3500.00, 'quantity' => 100]);
        $events[] = $event2;

        // Event 3: Upcoming event
        $event3 = Event::create([
            'name' => 'Tech Summit 2026',
            'description' => 'Join the top minds in AI, Web3, and emerging technologies. Network with industry leaders and gain insights into the future of tech.',
            'venue' => 'KICC, Nairobi',
            'event_date' => now()->addDays(30)->toDateString(),
            'start_time' => '08:00:00',
            'end_time' => '17:00:00',
            'status' => 'upcoming',
            'category' => 'Business & Corporate',
            'featured' => true,
            'max_capacity' => 1500,
            'enforce_capacity' => true,
            'created_by' => $creator2->id,
        ]);

        TicketCategory::create(['event_id' => $event3->id, 'name' => 'Early Bird', 'price' => 3000.00, 'quantity' => 300]);
        TicketCategory::create(['event_id' => $event3->id, 'name' => 'Student Pass', 'price' => 2000.00, 'quantity' => 200]);
        TicketCategory::create(['event_id' => $event3->id, 'name' => 'Regular Pass', 'price' => 6000.00, 'quantity' => 800]);
        TicketCategory::create(['event_id' => $event3->id, 'name' => 'Corporate Group (5+ tickets)', 'price' => 5000.00, 'quantity' => 200]);
        $events[] = $event3;

        // Event 4: Sports event
        $event4 = Event::create([
            'name' => 'Nairobi Marathon 2026',
            'description' => 'Annual charity marathon supporting local education initiatives. Multiple race categories available.',
            'venue' => 'Uhuru Gardens',
            'event_date' => now()->addDays(45)->toDateString(),
            'start_time' => '06:00:00',
            'end_time' => '12:00:00',
            'status' => 'upcoming',
            'category' => 'Sports & Fitness',
            'featured' => false,
            'max_capacity' => 2000,
            'enforce_capacity' => true,
            'created_by' => $creator1->id,
        ]);

        TicketCategory::create(['event_id' => $event4->id, 'name' => '5K Fun Run', 'price' => 800.00, 'quantity' => 500]);
        TicketCategory::create(['event_id' => $event4->id, 'name' => '10K Race', 'price' => 1200.00, 'quantity' => 800]);
        TicketCategory::create(['event_id' => $event4->id, 'name' => 'Half Marathon', 'price' => 2000.00, 'quantity' => 500]);
        TicketCategory::create(['event_id' => $event4->id, 'name' => 'Full Marathon', 'price' => 3000.00, 'quantity' => 200]);
        $events[] = $event4;

        // Event 5: Another upcoming event
        $event5 = Event::create([
            'name' => 'Food & Wine Festival',
            'description' => 'Experience the finest local and international cuisines with wine pairings from expert sommeliers.',
            'venue' => 'Karen Country Club',
            'event_date' => now()->addDays(60)->toDateString(),
            'start_time' => '12:00:00',
            'end_time' => '20:00:00',
            'status' => 'upcoming',
            'category' => 'Food & Drink',
            'featured' => true,
            'max_capacity' => 500,
            'enforce_capacity' => true,
            'created_by' => $creator2->id,
        ]);

        TicketCategory::create(['event_id' => $event5->id, 'name' => 'General Admission', 'price' => 4000.00, 'quantity' => 300]);
        TicketCategory::create(['event_id' => $event5->id, 'name' => 'Premium Tasting', 'price' => 8000.00, 'quantity' => 150]);
        $events[] = $event5;

        return $events;
    }

    protected function seedOrdersAndTickets(array $events, array $customers): void
    {
        $orderCounter = 1000;

        foreach ($events as $event) {
            $categories = TicketCategory::where('event_id', $event->id)->get();

            // Create multiple orders for each event
            $numOrders = rand(5, 15);

            for ($i = 0; $i < $numOrders; $i++) {
                $customer = $customers[array_rand($customers)];
                $category = $categories[array_rand($categories->toArray())];
                $quantity = rand(1, 4);
                $amount = $category->price * $quantity;

                // Randomly assign payment status (more paid for past events)
                $paymentStatuses = ['paid', 'paid', 'paid', 'pending', 'failed'];
                if ($event->status === 'completed') {
                    $paymentStatus = 'paid'; // All completed event orders are paid
                } else {
                    $paymentStatus = $paymentStatuses[array_rand($paymentStatuses)];
                }

                $order = Order::create([
                    'order_number' => 'ORD-' . str_pad($orderCounter++, 6, '0', STR_PAD_LEFT),
                    'user_id' => $customer->id,
                    'event_id' => $event->id,
                    'ticket_category_id' => $category->id,
                    'quantity' => $quantity,
                    'amount' => $amount,
                    'payment_status' => $paymentStatus,
                    'payment_reference' => $paymentStatus === 'paid' ? 'MPESA-' . strtoupper(Str::random(10)) : null,
                    'created_at' => $event->status === 'completed' ? now()->subDays(rand(30, 40)) : now()->subDays(rand(1, 10)),
                ]);

                // Create M-Pesa transaction for paid orders
                if ($paymentStatus === 'paid') {
                    MpesaTransaction::create([
                        'merchant_request_id' => 'MR-' . strtoupper(Str::random(10)),
                        'checkout_request_id' => 'CR-' . strtoupper(Str::random(10)),
                        'result_code' => '0',
                        'result_desc' => 'The service request is processed successfully.',
                        'amount' => $amount,
                        'mpesa_receipt_number' => 'MPESA-' . strtoupper(Str::random(10)),
                        'transaction_date' => $order->created_at->format('YmdHis'),
                        'phone_number' => $customer->phone,
                        'order_id' => $order->id,
                        'created_at' => $order->created_at,
                    ]);
                }

                // Create tickets for paid orders
                if ($paymentStatus === 'paid') {
                    for ($t = 0; $t < $quantity; $t++) {
                        $ticket = Ticket::create([
                            'order_id' => $order->id,
                            'ticket_category_id' => $category->id,
                            'qr_code' => strtoupper(Str::random(16)),
                            'uuid' => Str::uuid(),
                            'checksum' => hash('sha256', $order->id . '-' . $t . '-' . time()),
                            'is_checked_in' => false,
                            'created_at' => $order->created_at,
                        ]);
                    }
                }
            }
        }
    }

    protected function seedCheckIns(array $agents): void
    {
        // Get all tickets from completed and ongoing events
        $completedEvents = Event::whereIn('status', ['completed', 'ongoing'])->pluck('id');
        $tickets = Ticket::whereHas('order', function ($query) use ($completedEvents) {
            $query->whereIn('event_id', $completedEvents);
        })->get();

        // Check in 60-80% of tickets from completed/ongoing events
        $checkInPercentage = rand(60, 80);
        $numToCheckIn = (int) ($tickets->count() * ($checkInPercentage / 100));

        $ticketsToCheckIn = $tickets->random(min($numToCheckIn, $tickets->count()));

        foreach ($ticketsToCheckIn as $ticket) {
            $agent = $agents[array_rand($agents)];

            CheckIn::create([
                'ticket_id' => $ticket->id,
                'agent_id' => $agent->user_id,
                'created_at' => now()->subDays(rand(0, 30)),
            ]);

            $ticket->update([
                'is_checked_in' => true,
                'checked_in_at' => now()->subDays(rand(0, 30)),
            ]);
        }
    }
}
