<?php

namespace Database\Seeders;

use App\Models\Page;
use App\Models\User;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;

class PagesSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $this->command->info('Creating default pages...');

        // Get the first super admin user to be the creator
        $creator = User::whereHas('roles', function ($query) {
            $query->where('name', 'super_admin');
        })->first();

        if (!$creator) {
            $creator = User::first();
        }

        $pages = [
            [
                'title' => 'How It Works',
                'slug' => 'how-it-works',
                'content' => $this->getHowItWorksContent(),
                'excerpt' => 'Learn how to buy tickets and attend events on our platform',
                'status' => 'published',
                'template' => 'default',
                'meta' => json_encode([
                    'title' => 'How It Works - MATIKO',
                    'description' => 'Step-by-step guide on how to browse events, purchase tickets, and access your tickets.',
                    'keywords' => 'how it works, buy tickets, event tickets, guide',
                ]),
                'created_by' => $creator->id,
            ],
            [
                'title' => 'Help Center',
                'slug' => 'help-center',
                'content' => $this->getHelpCenterContent(),
                'excerpt' => 'Find answers to frequently asked questions',
                'status' => 'published',
                'template' => 'default',
                'meta' => json_encode([
                    'title' => 'Help Center - MATIKO',
                    'description' => 'Get help with tickets, payments, refunds, and more.',
                    'keywords' => 'help, faq, support, assistance',
                ]),
                'created_by' => $creator->id,
            ],
            [
                'title' => 'Contact Us',
                'slug' => 'contact-us',
                'content' => $this->getContactUsContent(),
                'excerpt' => 'Get in touch with our support team',
                'status' => 'published',
                'template' => 'default',
                'meta' => json_encode([
                    'title' => 'Contact Us - MATIKO',
                    'description' => 'Need help? Contact our support team for assistance.',
                    'keywords' => 'contact, support, help, customer service',
                ]),
                'created_by' => $creator->id,
            ],
            [
                'title' => 'About Us',
                'slug' => 'about-us',
                'content' => $this->getAboutUsContent(),
                'excerpt' => 'Learn more about our event ticketing platform',
                'status' => 'published',
                'template' => 'default',
                'meta' => json_encode([
                    'title' => 'About Us - MATIKO',
                    'description' => 'Learn about our mission to make event ticketing simple and accessible.',
                    'keywords' => 'about, company, mission, vision',
                ]),
                'created_by' => $creator->id,
            ],
            [
                'title' => 'Privacy Policy',
                'slug' => 'privacy-policy',
                'content' => $this->getPrivacyPolicyContent(),
                'excerpt' => 'How we collect, use, and protect your personal information',
                'status' => 'published',
                'template' => 'default',
                'meta' => json_encode([
                    'title' => 'Privacy Policy - MATIKO',
                    'description' => 'Our commitment to protecting your privacy and personal data.',
                    'keywords' => 'privacy, policy, data protection, gdpr',
                ]),
                'created_by' => $creator->id,
            ],
            [
                'title' => 'Terms of Service',
                'slug' => 'terms-of-service',
                'content' => $this->getTermsOfServiceContent(),
                'excerpt' => 'Terms and conditions for using our ticketing platform',
                'status' => 'published',
                'template' => 'default',
                'meta' => json_encode([
                    'title' => 'Terms of Service - MATIKO',
                    'description' => 'Terms and conditions governing the use of our platform.',
                    'keywords' => 'terms, conditions, legal, agreement',
                ]),
                'created_by' => $creator->id,
            ],
            [
                'title' => 'Refund Policy',
                'slug' => 'refund-policy',
                'content' => $this->getRefundPolicyContent(),
                'excerpt' => 'Our ticket refund and cancellation policy',
                'status' => 'published',
                'template' => 'default',
                'meta' => json_encode([
                    'title' => 'Refund Policy - MATIKO',
                    'description' => 'Learn about our refund policy for event tickets.',
                    'keywords' => 'refund, cancellation, policy, tickets',
                ]),
                'created_by' => $creator->id,
            ],
        ];

        foreach ($pages as $pageData) {
            Page::updateOrCreate(
                ['slug' => $pageData['slug']],
                $pageData
            );
            $this->command->info("✓ Created page: {$pageData['title']}");
        }

        $this->command->info('✓ Default pages created successfully!');
    }

    private function getHowItWorksContent(): string
    {
        return <<<HTML
<h2>Getting Started with MATIKO</h2>

<p>Welcome to MATIKO, your premier event ticketing platform. Here's how easy it is to get started:</p>

<h3>1. Browse Events</h3>
<p>Explore our wide range of events including concerts, conferences, sports, festivals, and more. Use our search and filter tools to find exactly what you're looking for.</p>

<h3>2. Select Your Tickets</h3>
<p>Choose your preferred event and select the number of tickets you need. You can see real-time availability and pricing for different ticket types.</p>

<h3>3. Secure Checkout</h3>
<p>Complete your purchase securely using M-Pesa or Airtel Money. We support guest checkout, so you don't need to create an account to buy tickets.</p>

<h3>4. Receive Your Tickets</h3>
<p>Get instant confirmation via email and SMS. Your tickets will include a unique QR code for entry verification.</p>

<h3>5. Attend the Event</h3>
<p>Show your QR code at the event entrance. Our agents will scan it for quick and seamless entry.</p>

<h3>Need Help?</h3>
<p>If you have any questions, visit our <a href="/page/help-center">Help Center</a> or <a href="/page/contact-us">Contact Us</a>.</p>
HTML;
    }

    private function getHelpCenterContent(): string
    {
        return <<<HTML
<h2>Frequently Asked Questions</h2>

<h3>How do I purchase tickets?</h3>
<p>Browse our events, select the event you want to attend, choose your tickets, and complete the checkout process using M-Pesa or Airtel Money.</p>

<h3>Do I need an account to buy tickets?</h3>
<p>No, we support guest checkout. However, creating an account makes it easier to track your orders and access your tickets.</p>

<h3>What payment methods do you accept?</h3>
<p>We currently accept M-Pesa and Airtel Money for all transactions.</p>

<h3>How will I receive my tickets?</h3>
<p>After successful payment, you'll receive your tickets via email and SMS. Each ticket contains a unique QR code.</p>

<h3>Can I get a refund?</h3>
<p>Refund policies vary by event. Please check our <a href="/page/refund-policy">Refund Policy</a> and the specific event terms.</p>

<h3>What if I lose my ticket?</h3>
<p>You can always access your tickets from the "My Tickets" section if you created an account. Otherwise, check your email or SMS for the ticket details.</p>

<h3>How do I contact support?</h3>
<p>You can reach us through our <a href="/page/contact-us">Contact Us</a> page, email us at info@matiko.com, or call +254700000000.</p>

<h3>Is my payment information secure?</h3>
<p>Yes, all payments are processed through secure mobile money platforms. We do not store your payment details.</p>

<h3>Can I transfer my ticket to someone else?</h3>
<p>This depends on the event organizer's policy. Please contact support for assistance with ticket transfers.</p>
HTML;
    }

    private function getContactUsContent(): string
    {
        return <<<HTML
<h2>Get in Touch</h2>

<p>We're here to help! Reach out to us through any of the following channels:</p>

<h3>Customer Support</h3>
<ul>
    <li><strong>Email:</strong> info@matiko.com</li>
    <li><strong>Phone:</strong> +254700000000</li>
    <li><strong>Hours:</strong> Monday - Sunday, 8:00 AM - 8:00 PM EAT</li>
</ul>

<h3>Office Location</h3>
<p>Nairobi, Kenya</p>

<h3>Before You Contact Us</h3>
<p>For quick answers, please check our <a href="/page/help-center">Help Center</a> where we've answered the most common questions.</p>

<h3>Event Organizers</h3>
<p>If you're interested in listing your event on our platform, please email us at events@matiko.com with details about your event.</p>

<h3>Partnerships & Business Inquiries</h3>
<p>For partnership opportunities and business inquiries, please contact business@matiko.com</p>
HTML;
    }

    private function getAboutUsContent(): string
    {
        return <<<HTML
<h2>About MATIKO</h2>

<p>MATIKO is Kenya's premier event ticketing platform, dedicated to making event discovery and ticket purchasing simple, secure, and accessible to everyone.</p>

<h3>Our Mission</h3>
<p>We believe that attending great events should be effortless. Our mission is to connect event organizers with attendees through a seamless, secure, and user-friendly platform.</p>

<h3>What We Do</h3>
<p>We provide a comprehensive ticketing solution that includes:</p>
<ul>
    <li>Easy online ticket purchasing with mobile money integration</li>
    <li>Secure QR code-based ticket verification</li>
    <li>Real-time event discovery and browsing</li>
    <li>Instant ticket delivery via email and SMS</li>
    <li>Dedicated customer support</li>
</ul>

<h3>Why Choose MATIKO?</h3>
<ul>
    <li><strong>Secure Payments:</strong> All transactions are processed through trusted mobile money platforms</li>
    <li><strong>Instant Tickets:</strong> Get your tickets immediately after purchase</li>
    <li><strong>Easy Access:</strong> No need to print - show your QR code from your phone</li>
    <li><strong>Local Support:</strong> Dedicated customer service team available when you need help</li>
    <li><strong>Wide Selection:</strong> From concerts to conferences, sports to festivals</li>
</ul>

<h3>Our Values</h3>
<ul>
    <li><strong>Trust:</strong> We protect your data and ensure secure transactions</li>
    <li><strong>Simplicity:</strong> Easy to use for everyone, from first-time users to regular attendees</li>
    <li><strong>Reliability:</strong> Tickets delivered on time, every time</li>
    <li><strong>Support:</strong> Always here to help when you need us</li>
</ul>
HTML;
    }

    private function getPrivacyPolicyContent(): string
    {
        return <<<HTML
<h2>Privacy Policy</h2>

<p><em>Last updated: January 2026</em></p>

<h3>1. Information We Collect</h3>
<p>We collect information you provide directly to us when you:</p>
<ul>
    <li>Create an account</li>
    <li>Purchase tickets</li>
    <li>Contact customer support</li>
    <li>Subscribe to our newsletter</li>
</ul>

<p>This information may include:</p>
<ul>
    <li>Name and contact information (email, phone number)</li>
    <li>Payment information (processed securely through mobile money providers)</li>
    <li>Event attendance history</li>
    <li>Communication preferences</li>
</ul>

<h3>2. How We Use Your Information</h3>
<p>We use the information we collect to:</p>
<ul>
    <li>Process your ticket purchases</li>
    <li>Send you tickets and order confirmations</li>
    <li>Provide customer support</li>
    <li>Send you event updates and promotions (with your consent)</li>
    <li>Improve our services</li>
    <li>Prevent fraud and ensure platform security</li>
</ul>

<h3>3. Information Sharing</h3>
<p>We may share your information with:</p>
<ul>
    <li>Event organizers (name and email for attendance purposes)</li>
    <li>Payment processors (to complete transactions)</li>
    <li>Service providers who assist in our operations</li>
    <li>Law enforcement when required by law</li>
</ul>

<p>We never sell your personal information to third parties.</p>

<h3>4. Data Security</h3>
<p>We implement appropriate technical and organizational measures to protect your personal information against unauthorized access, alteration, disclosure, or destruction.</p>

<h3>5. Your Rights</h3>
<p>You have the right to:</p>
<ul>
    <li>Access your personal information</li>
    <li>Correct inaccurate data</li>
    <li>Request deletion of your data</li>
    <li>Opt-out of marketing communications</li>
    <li>Withdraw consent for data processing</li>
</ul>

<h3>6. Cookies</h3>
<p>We use cookies and similar technologies to improve your experience, analyze site traffic, and personalize content.</p>

<h3>7. Contact Us</h3>
<p>For privacy-related questions or concerns, please contact us at privacy@matiko.com</p>
HTML;
    }

    private function getTermsOfServiceContent(): string
    {
        return <<<HTML
<h2>Terms of Service</h2>

<p><em>Last updated: January 2026</em></p>

<h3>1. Acceptance of Terms</h3>
<p>By accessing and using MATIKO, you agree to be bound by these Terms of Service. If you do not agree to these terms, please do not use our platform.</p>

<h3>2. Use of Service</h3>
<p>You may use MATIKO to:</p>
<ul>
    <li>Browse and search for events</li>
    <li>Purchase tickets for events</li>
    <li>Manage your ticket orders</li>
</ul>

<h3>3. Account Registration</h3>
<p>While account registration is optional, creating an account allows you to:</p>
<ul>
    <li>Track your order history</li>
    <li>Save payment preferences</li>
    <li>Receive personalized recommendations</li>
</ul>

<p>You are responsible for maintaining the confidentiality of your account credentials.</p>

<h3>4. Ticket Purchases</h3>
<ul>
    <li>All ticket sales are subject to availability</li>
    <li>Prices are displayed in Kenyan Shillings (KES)</li>
    <li>Payment must be completed within the specified time limit</li>
    <li>You will receive confirmation via email and SMS upon successful purchase</li>
</ul>

<h3>5. Refunds and Cancellations</h3>
<p>Refund policies are determined by individual event organizers. Please refer to our <a href="/page/refund-policy">Refund Policy</a> for more details.</p>

<h3>6. Prohibited Activities</h3>
<p>You may not:</p>
<ul>
    <li>Use the platform for fraudulent purposes</li>
    <li>Resell tickets at inflated prices (scalping)</li>
    <li>Share or transfer tickets in violation of event policies</li>
    <li>Attempt to circumvent security measures</li>
    <li>Use automated systems to purchase tickets (bots)</li>
</ul>

<h3>7. Intellectual Property</h3>
<p>All content on MATIKO, including logos, designs, text, and graphics, is protected by copyright and trademark laws.</p>

<h3>8. Limitation of Liability</h3>
<p>MATIKO is not responsible for:</p>
<ul>
    <li>Event cancellations or changes made by organizers</li>
    <li>The quality or safety of events</li>
    <li>Lost or stolen tickets</li>
    <li>Technical issues beyond our control</li>
</ul>

<h3>9. Changes to Terms</h3>
<p>We reserve the right to modify these terms at any time. Continued use of the platform constitutes acceptance of updated terms.</p>

<h3>10. Governing Law</h3>
<p>These terms are governed by the laws of Kenya.</p>

<h3>11. Contact</h3>
<p>For questions about these terms, contact us at legal@matiko.com</p>
HTML;
    }

    private function getRefundPolicyContent(): string
    {
        return <<<HTML
<h2>Refund Policy</h2>

<p><em>Last updated: January 2026</em></p>

<h3>General Policy</h3>
<p>Refunds for event tickets are subject to the individual event organizer's policies. MATIKO facilitates ticket sales on behalf of event organizers and processes refunds in accordance with their specific terms.</p>

<h3>Standard Refund Conditions</h3>

<h4>Event Cancellations</h4>
<p>If an event is officially cancelled by the organizer:</p>
<ul>
    <li>You are entitled to a full refund of your ticket price</li>
    <li>Refunds will be processed within 14 business days</li>
    <li>You will receive notification via email and SMS</li>
    <li>Refunds are issued to the original payment method (M-Pesa or Airtel Money)</li>
</ul>

<h4>Event Postponements</h4>
<p>If an event is postponed to a new date:</p>
<ul>
    <li>Your tickets remain valid for the new date</li>
    <li>If you cannot attend the new date, refund eligibility depends on the organizer's policy</li>
    <li>Contact support within 7 days of the postponement announcement</li>
</ul>

<h4>Customer-Initiated Cancellations</h4>
<p>For voluntary cancellations by ticket holders:</p>
<ul>
    <li>Refunds are only available if permitted by the event organizer</li>
    <li>Processing fees may apply</li>
    <li>Refund requests must be submitted at least 48 hours before the event</li>
    <li>Some events may have non-refundable tickets</li>
</ul>

<h3>How to Request a Refund</h3>
<ol>
    <li>Contact our support team at info@matiko.com or +254700000000</li>
    <li>Provide your order number and ticket details</li>
    <li>State the reason for your refund request</li>
    <li>Our team will verify eligibility with the event organizer</li>
    <li>You will receive a response within 48 hours</li>
</ol>

<h3>Processing Times</h3>
<ul>
    <li><strong>Approved Refunds:</strong> Processed within 7-14 business days</li>
    <li><strong>M-Pesa Refunds:</strong> Typically 3-5 business days</li>
    <li><strong>Airtel Money Refunds:</strong> Typically 3-5 business days</li>
</ul>

<h3>Non-Refundable Situations</h3>
<p>Refunds are generally not available for:</p>
<ul>
    <li>Change of mind after ticket purchase</li>
    <li>Failure to attend the event</li>
    <li>Lost or damaged tickets (replacements may be available)</li>
    <li>Events that occur as scheduled</li>
    <li>Tickets marked as "non-refundable" at time of purchase</li>
</ul>

<h3>Exceptions and Special Cases</h3>
<p>In exceptional circumstances such as medical emergencies or force majeure events, we may work with event organizers to accommodate refund requests. Each case is evaluated individually.</p>

<h3>Contact Us</h3>
<p>For refund inquiries, please contact:</p>
<ul>
    <li>Email: refunds@matiko.com</li>
    <li>Phone: +254700000000</li>
    <li>Support hours: Monday - Sunday, 8:00 AM - 8:00 PM EAT</li>
</ul>
HTML;
    }
}
