import Alpine from 'alpinejs';
import focus from '@alpinejs/focus';
import intersect from '@alpinejs/intersect';
import collapse from '@alpinejs/collapse';

// Register Alpine plugins
Alpine.plugin(focus);
Alpine.plugin(intersect);
Alpine.plugin(collapse);

// Global Alpine.js data/stores
Alpine.data('mobileMenu', () => ({
    open: false,
    toggle() {
        this.open = !this.open;
    },
    close() {
        this.open = false;
    }
}));

Alpine.data('flashMessage', () => ({
    show: true,
    init() {
        // Auto-hide after 5 seconds
        setTimeout(() => {
            this.show = false;
        }, 5000);
    },
    close() {
        this.show = false;
    }
}));

Alpine.data('imageLoader', () => ({
    loaded: false,
    error: false,
    onLoad() {
        this.loaded = true;
    },
    onError() {
        this.error = true;
    }
}));

Alpine.data('countdown', (targetDate) => ({
    days: 0,
    hours: 0,
    minutes: 0,
    seconds: 0,
    expired: false,

    init() {
        this.updateCountdown();
        setInterval(() => {
            this.updateCountdown();
        }, 1000);
    },

    updateCountdown() {
        const now = new Date().getTime();
        const target = new Date(targetDate).getTime();
        const distance = target - now;

        if (distance < 0) {
            this.expired = true;
            return;
        }

        this.days = Math.floor(distance / (1000 * 60 * 60 * 24));
        this.hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
        this.minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
        this.seconds = Math.floor((distance % (1000 * 60)) / 1000);
    }
}));

// Scroll animations using Intersection Observer
document.addEventListener('alpine:init', () => {
    Alpine.directive('scroll-reveal', (el, { expression }, { evaluate }) => {
        const options = {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    el.classList.add('animate-slide-up');
                    observer.unobserve(el);
                }
            });
        }, options);

        observer.observe(el);
    });
});

// Global helpers
window.formatCurrency = (amount) => {
    return new Intl.NumberFormat('en-KE', {
        style: 'currency',
        currency: 'KES',
        minimumFractionDigits: 0,
        maximumFractionDigits: 0
    }).format(amount);
};

window.formatDate = (dateString) => {
    const date = new Date(dateString);
    return new Intl.DateTimeFormat('en-KE', {
        weekday: 'long',
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    }).format(date);
};

window.formatTime = (timeString) => {
    const [hours, minutes] = timeString.split(':');
    const date = new Date();
    date.setHours(hours, minutes);
    return new Intl.DateTimeFormat('en-KE', {
        hour: 'numeric',
        minute: '2-digit',
        hour12: true
    }).format(date);
};

// Start Alpine after DOM is loaded
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        Alpine.start();
        console.log('MATIKO Ticketing Platform - Powered by Alpine.js 🎟️');
    });
} else {
    Alpine.start();
    console.log('MATIKO Ticketing Platform - Powered by Alpine.js 🎟️');
}
