@extends('layout')

@section('content')
<div class="max-w-2xl mx-auto text-center pt-12">
    <div class="bg-white rounded-2xl shadow-xl p-12">
        <!-- Processing Animation -->
        <div id="processing-section">
            <div class="flex justify-center mb-6">
                <div class="rounded-full bg-blue-100 p-4 animate-pulse">
                    <svg class="w-16 h-16 text-blue-600 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                    </svg>
                </div>
            </div>

            <h1 class="text-3xl font-bold text-gray-900 mb-4">Processing Payment</h1>
            <p class="text-xl text-gray-600 mb-4">Please check your phone <strong>{{ $order->phone_number }}</strong> and enter your M-Pesa PIN.</p>
            <p class="text-gray-500 mb-8">We're waiting for payment confirmation...</p>

            <div class="bg-gray-50 rounded-lg p-6 mb-8 text-left">
                <h3 class="font-bold text-gray-800 mb-2">Order #{{ $order->order_number }}</h3>
                <p class="text-gray-600 mb-4">We'll notify you once payment is confirmed:</p>
                <ul class="space-y-2 text-sm text-gray-700">
                    @if($order->user->email)
                    <li class="flex items-center">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"></path>
                        </svg>
                        Email: {{ $order->user->email }}
                    </li>
                    @endif
                    <li class="flex items-center">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 18h.01M8 21h8a2 2 0 002-2V5a2 2 0 00-2-2H8a2 2 0 00-2 2v14a2 2 0 002 2z"></path>
                        </svg>
                        SMS: {{ $order->phone_number }}
                    </li>
                </ul>
            </div>

            <!-- Debug Info -->
            <div class="bg-gray-100 rounded-lg p-4 mb-6 text-left text-xs">
                <p><strong>Debug Info:</strong></p>
                <p>Order ID: {{ $order->id }}</p>
                <p>Status: <span id="current-status">checking...</span></p>
                <p>Polls: <span id="poll-count">0</span></p>
                <div class="mt-2 space-x-2">
                    <button onclick="simulateSuccess()" class="bg-green-500 text-white px-2 py-1 rounded text-xs">Test Success</button>
                    <button onclick="simulateFailure()" class="bg-red-500 text-white px-2 py-1 rounded text-xs">Test Failure</button>
                    <button onclick="checkStatus()" class="bg-blue-500 text-white px-2 py-1 rounded text-xs">Manual Check</button>
                </div>
            </div>

        </div>

        <!-- Success Section (hidden initially) -->
        <div id="success-section" style="display: none;">
            <div class="flex justify-center mb-6">
                <div class="rounded-full bg-green-100 p-4">
                    <svg class="w-16 h-16 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                    </svg>
                </div>
            </div>

            <h1 class="text-3xl font-bold text-gray-900 mb-4">Payment Successful!</h1>
            <p id="success-message" class="text-xl text-gray-600 mb-8">Your ticket has been generated and sent to your email/SMS.</p>

            <!-- Download Options -->
            <div class="mb-6">
                <h4 class="font-bold text-gray-800 mb-3 text-center">Download Options</h4>

                <!-- Download All Tickets Button -->
                <div class="flex gap-4 justify-center mb-4">
                    <a id="download-all-link" href="#" class="bg-green-600 hover:bg-green-700 text-white font-bold py-3 px-6 rounded-lg shadow-lg hover:shadow-xl transition">
                        <span id="download-all-text">Download All Tickets</span>
                    </a>
                </div>

                <!-- Individual Tickets Section (shown only for multiple tickets) -->
                <div id="individual-tickets-section" style="display: none;" class="bg-white rounded-lg p-4 border border-gray-200">
                    <h5 class="font-semibold text-gray-700 mb-3 text-sm">Or download individual tickets:</h5>
                    <div id="individual-tickets-list" class="space-y-2">
                        <!-- Individual ticket links will be loaded here -->
                    </div>
                </div>
            </div>

            <div class="flex gap-4 justify-center">
                <a href="/" class="bg-gray-600 hover:bg-gray-700 text-white font-bold py-3 px-6 rounded-lg">
                    Return to Home
                </a>
            </div>
        </div>

        <!-- Failure Section (hidden initially) -->
        <div id="failure-section" style="display: none;">
            <div class="flex justify-center mb-6">
                <div class="rounded-full bg-red-100 p-4">
                    <svg class="w-16 h-16 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </div>
            </div>

            <h1 class="text-3xl font-bold text-gray-900 mb-4">Payment Failed</h1>
            <p class="text-xl text-gray-600 mb-8">Unfortunately, your payment could not be processed.</p>

            <div class="bg-red-50 rounded-lg p-6 mb-8 text-left">
                <h3 class="font-bold text-red-800 mb-2">Order #{{ $order->order_number }}</h3>
                <p class="text-red-700">Please try again or contact support if the issue persists.</p>
            </div>

            <div class="flex gap-4 justify-center">
                <a href="{{ route('checkout', $order->event_id) }}" class="bg-red-600 hover:bg-red-700 text-white font-bold py-3 px-6 rounded-lg">
                    Try Again
                </a>
                <a href="/" class="bg-gray-600 hover:bg-gray-700 text-white font-bold py-3 px-6 rounded-lg">
                    Return to Home
                </a>
            </div>
        </div>

        <!-- Timeout Section (hidden initially) -->
        <div id="timeout-section" style="display: none;">
            <div class="flex justify-center mb-6">
                <div class="rounded-full bg-yellow-100 p-4">
                    <svg class="w-16 h-16 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
                    </svg>
                </div>
            </div>

            <h1 class="text-3xl font-bold text-gray-900 mb-4">Payment Processing Timeout</h1>
            <p class="text-xl text-gray-600 mb-8">We're still waiting for payment confirmation from M-Pesa.</p>

            <div class="bg-yellow-50 rounded-lg p-6 mb-8 text-left">
                <h3 class="font-bold text-yellow-800 mb-2">Order #{{ $order->order_number }}</h3>
                <p class="text-yellow-700 mb-4">This can happen due to network issues or delays in M-Pesa processing.</p>
                <p class="text-yellow-700">Please check your M-Pesa messages or contact our support team for assistance.</p>
            </div>

            <div class="flex gap-4 justify-center">
                <a href="{{ route('checkout', $order->event_id) }}" class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-3 px-6 rounded-lg">
                    Try New Payment
                </a>
                <a href="mailto:support@matiko.com" class="bg-gray-600 hover:bg-gray-700 text-white font-bold py-3 px-6 rounded-lg">
                    Contact Support
                </a>
            </div>
        </div>
    </div>
</div>

<script>
let statusCheckInterval;
let timeoutTimer;
let pollCount = 0;
const MAX_POLLS = 60; // 5 minutes (60 * 5 seconds)

function checkStatus() {
    console.log('Checking order status... (poll #' + (pollCount + 1) + ')');
    fetch(`/api/v1/orders/{{ $order->id }}/status`)
        .then(response => response.json())
        .then(data => {
            console.log('Status response:', data);
            pollCount++;

            // Update debug info
            document.getElementById('current-status').textContent = data.status;
            document.getElementById('poll-count').textContent = pollCount;

            if (data.status === 'paid') {
                console.log('Payment successful, showing success screen');
                showSuccess(data.ticket);
            } else if (data.status === 'failed') {
                console.log('Payment failed, showing failure screen');
                showFailure();
            } else if (pollCount >= MAX_POLLS) {
                console.log('Payment timeout reached, showing timeout screen');
                showTimeout();
            } else {
                console.log('Payment still pending:', data.status);
            }
            // If still pending and not timed out, continue polling
        })
        .catch(error => {
            console.error('Error checking status:', error);
            pollCount++;
            document.getElementById('poll-count').textContent = pollCount;
            if (pollCount >= MAX_POLLS) {
                showTimeout();
            }
        });
}

function simulateSuccess() {
    console.log('Simulating successful payment...');

    // Call the backend endpoint to actually process the payment
    fetch(`/api/v1/orders/{{ $order->id }}/simulate-success`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
        }
    })
    .then(response => response.json())
    .then(data => {
        console.log('Success response:', data);
        if (data.status === 'paid' && data.ticket) {
            showSuccess(data.ticket);
        } else {
            console.error('Unexpected response format:', data);
            alert('Success simulation completed but response format unexpected. Check console.');
        }
    })
    .catch(error => {
        console.error('Error simulating success:', error);
        alert('Failed to simulate success: ' + error.message);
    });
}

function simulateFailure() {
    console.log('Simulating failed payment...');

    // Call the backend endpoint to actually record the failure
    fetch(`/api/v1/orders/{{ $order->id }}/simulate-failure`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
        }
    })
    .then(response => response.json())
    .then(data => {
        console.log('Failure response:', data);
        if (data.status === 'failed') {
            showFailure();
        } else {
            console.error('Unexpected response format:', data);
            alert('Failure simulation completed but response format unexpected. Check console.');
        }
    })
    .catch(error => {
        console.error('Error simulating failure:', error);
        alert('Failed to simulate failure: ' + error.message);
    });
}

function showSuccess(ticket) {
    clearInterval(statusCheckInterval);
    clearTimeout(timeoutTimer);
    document.getElementById('processing-section').style.display = 'none';
    document.getElementById('success-section').style.display = 'block';

    if (ticket) {
        const ticketCount = ticket.ticket_count || 1;
        const isMultiple = ticketCount > 1;

        // Update messages based on ticket count
        document.getElementById('success-message').textContent =
            `Your ${isMultiple ? ticketCount + ' tickets have' : 'ticket has'} been generated and sent to your email/SMS.`;

        document.getElementById('download-all-text').textContent =
            isMultiple ? `Download All ${ticketCount} Tickets (PDF)` : 'Download Ticket';

        // Set download all tickets link
        document.getElementById('download-all-link').href = ticket.download_url || `/tickets/download/order-{{ $order->order_number }}.pdf`;

        // Show individual tickets section if multiple tickets
        if (isMultiple && ticket.tickets && ticket.tickets.length > 0) {
            document.getElementById('individual-tickets-section').style.display = 'block';

            let individualTicketsHtml = '';
            ticket.tickets.forEach((t, index) => {
                individualTicketsHtml += `
                    <a href="${t.download_url}"
                       class="flex items-center justify-between p-3 bg-gray-50 hover:bg-gray-100 rounded border border-gray-200 hover:border-green-500 transition">
                        <span class="font-medium text-gray-700">
                            Ticket ${index + 1} - ${t.qr_code}
                        </span>
                        <span class="text-green-600 font-semibold text-sm">Download</span>
                    </a>
                `;
            });
            document.getElementById('individual-tickets-list').innerHTML = individualTicketsHtml;
        }
    }
}

function showFailure() {
    clearInterval(statusCheckInterval);
    clearTimeout(timeoutTimer);
    document.getElementById('processing-section').style.display = 'none';
    document.getElementById('failure-section').style.display = 'block';
}

function showTimeout() {
    clearInterval(statusCheckInterval);
    clearTimeout(timeoutTimer);
    document.getElementById('processing-section').style.display = 'none';
    document.getElementById('timeout-section').style.display = 'block';
}

// Start polling for status updates
statusCheckInterval = setInterval(checkStatus, 5000); // Check every 5 seconds
checkStatus(); // Check immediately

// Safety timeout (5 minutes)
timeoutTimer = setTimeout(() => {
    console.log('Safety timeout reached');
    showTimeout();
}, 300000); // 5 minutes
</script>
@endsection
