<?php

namespace Tests\Feature;

use App\Models\Event;
use App\Models\Order;
use App\Models\Ticket;
use App\Models\TicketCategory;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class EventCapacityTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Event $event;
    protected TicketCategory $category;

    protected function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->event = Event::factory()->create([
            'name' => 'Limited Capacity Event',
            'venue' => 'Small Venue',
            'event_date' => now()->addDays(30),
            'status' => 'active',
            'max_capacity' => 50,
            'enforce_capacity' => true,
        ]);

        $this->category = TicketCategory::create([
            'event_id' => $this->event->id,
            'name' => 'General Admission',
            'price' => 1000.00,
            'quantity' => 100, // More than event capacity
        ]);
    }

    /** @test */
    public function it_can_calculate_total_tickets_sold()
    {
        // Create 5 orders with 2 tickets each = 10 tickets total
        for ($i = 0; $i < 5; $i++) {
            $order = Order::create([
                'order_number' => 'ORD-TEST' . $i,
                'user_id' => $this->user->id,
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'amount' => 1000,
                'quantity' => 2,
                'total_amount' => 2000,
                'payment_status' => 'paid',
                'phone_number' => '0712345678',
            ]);

            // Create 2 tickets for each order
            Ticket::factory()->count(2)->create([
                'order_id' => $order->id,
                'ticket_category_id' => $this->category->id,
            ]);
        }

        $this->assertEquals(10, $this->event->totalTicketsSold());
    }

    /** @test */
    public function it_can_calculate_remaining_capacity()
    {
        // Create 30 tickets
        $order = Order::create([
            'order_number' => 'ORD-TEST',
            'user_id' => $this->user->id,
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'amount' => 1000,
            'quantity' => 30,
            'total_amount' => 30000,
            'payment_status' => 'paid',
            'phone_number' => '0712345678',
        ]);

        Ticket::factory()->count(30)->create([
            'order_id' => $order->id,
            'ticket_category_id' => $this->category->id,
        ]);

        // Max capacity is 50, 30 sold, so 20 remaining
        $this->assertEquals(20, $this->event->remainingCapacity());
    }

    /** @test */
    public function it_detects_when_event_is_at_capacity()
    {
        // Create exactly 50 tickets (at capacity)
        $order = Order::create([
            'order_number' => 'ORD-TEST',
            'user_id' => $this->user->id,
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'amount' => 1000,
            'quantity' => 50,
            'total_amount' => 50000,
            'payment_status' => 'paid',
            'phone_number' => '0712345678',
        ]);

        Ticket::factory()->count(50)->create([
            'order_id' => $order->id,
            'ticket_category_id' => $this->category->id,
        ]);

        $this->assertTrue($this->event->isAtCapacity());
    }

    /** @test */
    public function it_detects_when_event_is_not_at_capacity()
    {
        // Create 40 tickets (10 under capacity)
        $order = Order::create([
            'order_number' => 'ORD-TEST',
            'user_id' => $this->user->id,
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'amount' => 1000,
            'quantity' => 40,
            'total_amount' => 40000,
            'payment_status' => 'paid',
            'phone_number' => '0712345678',
        ]);

        Ticket::factory()->count(40)->create([
            'order_id' => $order->id,
            'ticket_category_id' => $this->category->id,
        ]);

        $this->assertFalse($this->event->isAtCapacity());
    }

    /** @test */
    public function it_validates_if_quantity_can_be_purchased()
    {
        // Create 45 tickets (5 remaining)
        $order = Order::create([
            'order_number' => 'ORD-TEST',
            'user_id' => $this->user->id,
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'amount' => 1000,
            'quantity' => 45,
            'total_amount' => 45000,
            'payment_status' => 'paid',
            'phone_number' => '0712345678',
        ]);

        Ticket::factory()->count(45)->create([
            'order_id' => $order->id,
            'ticket_category_id' => $this->category->id,
        ]);

        // Can purchase 5 tickets
        $this->assertTrue($this->event->canPurchaseQuantity(5));

        // Cannot purchase 6 tickets
        $this->assertFalse($this->event->canPurchaseQuantity(6));
    }

    /** @test */
    public function it_blocks_order_when_event_is_at_capacity()
    {
        // Fill the event to capacity
        $order = Order::create([
            'order_number' => 'ORD-TEST',
            'user_id' => $this->user->id,
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'amount' => 1000,
            'quantity' => 50,
            'total_amount' => 50000,
            'payment_status' => 'paid',
            'phone_number' => '0712345678',
        ]);

        Ticket::factory()->count(50)->create([
            'order_id' => $order->id,
            'ticket_category_id' => $this->category->id,
        ]);

        // Try to create another order
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 1,
            ]);

        $response->assertStatus(422);
        $response->assertJson([
            'message' => 'Event is at full capacity',
        ]);
    }

    /** @test */
    public function it_blocks_order_when_quantity_exceeds_remaining_capacity()
    {
        // Create 48 tickets (2 remaining)
        $order = Order::create([
            'order_number' => 'ORD-TEST',
            'user_id' => $this->user->id,
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'amount' => 1000,
            'quantity' => 48,
            'total_amount' => 48000,
            'payment_status' => 'paid',
            'phone_number' => '0712345678',
        ]);

        Ticket::factory()->count(48)->create([
            'order_id' => $order->id,
            'ticket_category_id' => $this->category->id,
        ]);

        // Try to purchase 3 tickets when only 2 remain
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 3,
            ]);

        $response->assertStatus(422);
        $response->assertJsonFragment([
            'remaining_capacity' => 2,
        ]);
    }

    /** @test */
    public function it_allows_orders_when_capacity_is_not_enforced()
    {
        // Disable capacity enforcement
        $this->event->update(['enforce_capacity' => false]);

        // Fill beyond capacity
        $order = Order::create([
            'order_number' => 'ORD-TEST',
            'user_id' => $this->user->id,
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'amount' => 1000,
            'quantity' => 50,
            'total_amount' => 50000,
            'payment_status' => 'paid',
            'phone_number' => '0712345678',
        ]);

        Ticket::factory()->count(50)->create([
            'order_id' => $order->id,
            'ticket_category_id' => $this->category->id,
        ]);

        // Should allow more orders even though at capacity
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 5,
            ]);

        $response->assertStatus(201);
    }

    /** @test */
    public function it_allows_unlimited_tickets_when_no_max_capacity_is_set()
    {
        // Create event without max capacity
        $unlimitedEvent = Event::factory()->create([
            'name' => 'Unlimited Event',
            'max_capacity' => null,
            'enforce_capacity' => true,
        ]);

        $category = TicketCategory::create([
            'event_id' => $unlimitedEvent->id,
            'name' => 'General',
            'price' => 1000,
            'quantity' => 1000,
        ]);

        // Should allow large orders
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $unlimitedEvent->id,
                'ticket_category_id' => $category->id,
                'phone_number' => '0712345678',
                'quantity' => 10,
            ]);

        $response->assertStatus(201);
        $this->assertNull($unlimitedEvent->remainingCapacity());
        $this->assertFalse($unlimitedEvent->isAtCapacity());
    }
}
