<?php

namespace Tests\Feature;

use App\Models\Event;
use App\Models\TicketCategory;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class GuestCheckoutTest extends TestCase
{
    use RefreshDatabase;

    protected Event $event;
    protected TicketCategory $category;

    protected function setUp(): void
    {
        parent::setUp();

        $this->event = Event::factory()->create([
            'name' => 'Public Event',
            'venue' => 'Test Venue',
            'event_date' => now()->addDays(30),
            'status' => 'active',
        ]);

        $this->category = TicketCategory::create([
            'event_id' => $this->event->id,
            'name' => 'General Admission',
            'price' => 1000.00,
            'quantity' => 100,
        ]);
    }

    /** @test */
    public function guest_can_create_order_without_authentication()
    {
        $response = $this->postJson('/api/orders', [
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'phone_number' => '0712345678',
            'email' => 'guest@example.com',
            'full_name' => 'John Guest',
            'quantity' => 2,
        ]);

        $response->assertStatus(201);
        $response->assertJsonStructure([
            'message',
            'order' => [
                'id',
                'order_number',
                'user_id',
                'event_id',
                'amount',
                'quantity',
                'total_amount',
            ],
        ]);
    }

    /** @test */
    public function it_creates_guest_user_account_automatically()
    {
        $guestEmail = 'newguest@example.com';

        $this->assertDatabaseMissing('users', [
            'email' => $guestEmail,
        ]);

        $response = $this->postJson('/api/orders', [
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'phone_number' => '0712345678',
            'email' => $guestEmail,
            'full_name' => 'New Guest User',
            'quantity' => 1,
        ]);

        $response->assertStatus(201);

        $this->assertDatabaseHas('users', [
            'email' => $guestEmail,
            'is_guest' => true,
        ]);
    }

    /** @test */
    public function it_reuses_existing_guest_account_by_email()
    {
        $guestEmail = 'existing@example.com';

        // Create a guest user first
        $existingGuest = User::create([
            'name' => 'Existing Guest',
            'email' => $guestEmail,
            'phone' => '0711111111',
            'is_guest' => true,
            'password' => bcrypt('random'),
        ]);

        // Make order with same email
        $response = $this->postJson('/api/orders', [
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'phone_number' => '0722222222',
            'email' => $guestEmail,
            'full_name' => 'Same Email Guest',
            'quantity' => 1,
        ]);

        $response->assertStatus(201);

        // Should use existing guest user
        $response->assertJsonPath('order.user_id', $existingGuest->id);

        // Should only have one user with this email
        $this->assertEquals(1, User::where('email', $guestEmail)->count());
    }

    /** @test */
    public function guest_order_requires_email()
    {
        $response = $this->postJson('/api/orders', [
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'phone_number' => '0712345678',
            'full_name' => 'John Guest',
            // Missing email
            'quantity' => 1,
        ]);

        $response->assertStatus(422);
        $response->assertJsonValidationErrors('email');
    }

    /** @test */
    public function guest_order_requires_full_name()
    {
        $response = $this->postJson('/api/orders', [
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'phone_number' => '0712345678',
            'email' => 'guest@example.com',
            // Missing full_name
            'quantity' => 1,
        ]);

        $response->assertStatus(422);
        $response->assertJsonValidationErrors('full_name');
    }

    /** @test */
    public function guest_order_requires_valid_email_format()
    {
        $response = $this->postJson('/api/orders', [
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'phone_number' => '0712345678',
            'email' => 'invalid-email-format',
            'full_name' => 'John Guest',
            'quantity' => 1,
        ]);

        $response->assertStatus(422);
        $response->assertJsonValidationErrors('email');
    }

    /** @test */
    public function authenticated_user_can_still_place_orders()
    {
        $user = User::factory()->create([
            'is_guest' => false,
        ]);

        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 1,
            ]);

        $response->assertStatus(201);
        $response->assertJsonPath('order.user_id', $user->id);
    }

    /** @test */
    public function guest_user_has_is_guest_flag_set_to_true()
    {
        $response = $this->postJson('/api/orders', [
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'phone_number' => '0712345678',
            'email' => 'flag-test@example.com',
            'full_name' => 'Flag Test User',
            'quantity' => 1,
        ]);

        $response->assertStatus(201);

        $user = User::where('email', 'flag-test@example.com')->first();
        $this->assertTrue($user->is_guest);
    }

    /** @test */
    public function guest_users_get_auto_generated_password()
    {
        $response = $this->postJson('/api/orders', [
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'phone_number' => '0712345678',
            'email' => 'password-test@example.com',
            'full_name' => 'Password Test User',
            'quantity' => 1,
        ]);

        $response->assertStatus(201);

        $user = User::where('email', 'password-test@example.com')->first();

        // Password should be set
        $this->assertNotNull($user->password);

        // Password should be hashed (bcrypt produces 60 character hash)
        $this->assertEquals(60, strlen($user->password));
    }

    /** @test */
    public function guest_order_stores_phone_number()
    {
        $phone = '0723456789';

        $response = $this->postJson('/api/orders', [
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'phone_number' => $phone,
            'email' => 'phone-test@example.com',
            'full_name' => 'Phone Test User',
            'quantity' => 1,
        ]);

        $response->assertStatus(201);

        $user = User::where('email', 'phone-test@example.com')->first();
        $this->assertEquals($phone, $user->phone);
    }

    /** @test */
    public function authenticated_user_order_does_not_require_email_or_name()
    {
        $user = User::factory()->create();

        // Should work without email or full_name
        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 1,
            ]);

        $response->assertStatus(201);
    }
}
