<?php

namespace Tests\Feature;

use App\Models\Event;
use App\Models\Order;
use App\Models\TicketCategory;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class MultipleTicketsPerOrderTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Event $event;
    protected TicketCategory $category;

    protected function setUp(): void
    {
        parent::setUp();

        // Create test user
        $this->user = User::factory()->create();

        // Create test event
        $this->event = Event::factory()->create([
            'name' => 'Test Concert',
            'venue' => 'Test Venue',
            'event_date' => now()->addDays(30),
            'status' => 'active',
        ]);

        // Create ticket category
        $this->category = TicketCategory::create([
            'event_id' => $this->event->id,
            'name' => 'General Admission',
            'price' => 1000.00,
            'quantity' => 100,
        ]);
    }

    /** @test */
    public function it_can_create_order_with_single_ticket()
    {
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 1,
            ]);

        $response->assertStatus(201);

        $this->assertDatabaseHas('orders', [
            'event_id' => $this->event->id,
            'user_id' => $this->user->id,
            'quantity' => 1,
            'total_amount' => 1000.00,
        ]);
    }

    /** @test */
    public function it_can_create_order_with_multiple_tickets()
    {
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 5,
            ]);

        $response->assertStatus(201);

        $this->assertDatabaseHas('orders', [
            'event_id' => $this->event->id,
            'user_id' => $this->user->id,
            'quantity' => 5,
            'total_amount' => 5000.00, // 5 * 1000
        ]);
    }

    /** @test */
    public function it_calculates_total_amount_correctly()
    {
        $quantity = 3;
        $expectedTotal = $this->category->price * $quantity;

        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => $quantity,
            ]);

        $response->assertStatus(201);
        $response->assertJsonPath('order.total_amount', $expectedTotal);
    }

    /** @test */
    public function it_validates_minimum_quantity()
    {
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 0, // Invalid - less than minimum
            ]);

        $response->assertStatus(422);
        $response->assertJsonValidationErrors('quantity');
    }

    /** @test */
    public function it_validates_maximum_quantity()
    {
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 11, // Invalid - more than maximum of 10
            ]);

        $response->assertStatus(422);
        $response->assertJsonValidationErrors('quantity');
    }

    /** @test */
    public function it_generates_multiple_tickets_after_payment()
    {
        // Create a paid order
        $order = Order::create([
            'order_number' => 'ORD-TEST123',
            'user_id' => $this->user->id,
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'amount' => $this->category->price,
            'quantity' => 3,
            'total_amount' => $this->category->price * 3,
            'payment_status' => 'paid',
            'phone_number' => '0712345678',
        ]);

        // Generate tickets
        $ticketService = app(\App\Services\TicketService::class);
        $tickets = $ticketService->generateTickets($order);

        // Assert 3 tickets were created
        $this->assertCount(3, $tickets);
        $this->assertEquals(3, $order->tickets()->count());

        // Assert each ticket has unique QR code and UUID
        $qrCodes = $tickets->pluck('qr_code')->toArray();
        $uuids = $tickets->pluck('uuid')->toArray();

        $this->assertCount(3, array_unique($qrCodes));
        $this->assertCount(3, array_unique($uuids));
    }

    /** @test */
    public function it_defaults_to_quantity_one_when_not_specified()
    {
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                // No quantity specified
            ]);

        $response->assertStatus(201);
        $response->assertJsonPath('order.quantity', 1);
    }

    /** @test */
    public function it_passes_correct_amount_to_payment_gateway()
    {
        $quantity = 4;
        $expectedTotal = $this->category->price * $quantity;

        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => $quantity,
            ]);

        // Note: This will fail if Daraja service is not configured
        // In production, you'd mock the Daraja service
        $response->assertStatus(201);

        $order = Order::where('user_id', $this->user->id)->latest()->first();
        $this->assertEquals($expectedTotal, $order->total_amount);
    }
}
