<?php

namespace Tests\Feature;

use App\Models\Event;
use App\Models\Order;
use App\Models\Ticket;
use App\Models\TicketCategory;
use App\Models\User;
use App\Services\TicketService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class QRCodeValidationTest extends TestCase
{
    use RefreshDatabase;

    protected TicketService $ticketService;
    protected User $user;
    protected Event $event;
    protected TicketCategory $category;
    protected Order $order;
    protected Ticket $ticket;

    protected function setUp(): void
    {
        parent::setUp();

        $this->ticketService = app(TicketService::class);
        $this->user = User::factory()->create();

        $this->event = Event::factory()->create([
            'name' => 'QR Test Event',
            'event_date' => now()->addDays(30),
        ]);

        $this->category = TicketCategory::create([
            'event_id' => $this->event->id,
            'name' => 'General',
            'price' => 1000,
            'quantity' => 100,
        ]);

        $this->order = Order::create([
            'order_number' => 'ORD-TEST123',
            'user_id' => $this->user->id,
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'amount' => 1000,
            'quantity' => 1,
            'total_amount' => 1000,
            'payment_status' => 'paid',
            'phone_number' => '0712345678',
        ]);

        $this->ticket = Ticket::factory()->create([
            'order_id' => $this->order->id,
            'ticket_category_id' => $this->category->id,
            'qr_code' => 'TKT-' . $this->order->id . '-TEST',
            'uuid' => \Illuminate\Support\Str::uuid(),
            'is_checked_in' => false,
        ]);

        // Generate checksum for the ticket
        $checksumReflection = new \ReflectionMethod($this->ticketService, 'generateChecksum');
        $checksumReflection->setAccessible(true);
        $checksum = $checksumReflection->invoke($this->ticketService, $this->ticket);
        $this->ticket->update(['checksum' => $checksum]);
    }

    /** @test */
    public function it_validates_correct_qr_code()
    {
        $qrData = json_encode([
            'ticket_id' => $this->ticket->uuid,
            'event_id' => $this->event->id,
            'checksum' => $this->ticket->checksum,
        ]);

        $result = $this->ticketService->validateTicket($qrData);

        $this->assertTrue($result['valid']);
        $this->assertEquals($this->ticket->id, $result['ticket']->id);
    }

    /** @test */
    public function it_rejects_invalid_json_format()
    {
        $qrData = 'invalid-json-string';

        $result = $this->ticketService->validateTicket($qrData);

        $this->assertFalse($result['valid']);
        $this->assertEquals('Invalid QR code format', $result['message']);
    }

    /** @test */
    public function it_rejects_missing_required_fields()
    {
        // Missing checksum field
        $qrData = json_encode([
            'ticket_id' => $this->ticket->uuid,
            'event_id' => $this->event->id,
        ]);

        $result = $this->ticketService->validateTicket($qrData);

        $this->assertFalse($result['valid']);
        $this->assertEquals('Invalid QR code format', $result['message']);
    }

    /** @test */
    public function it_rejects_non_existent_ticket()
    {
        $qrData = json_encode([
            'ticket_id' => \Illuminate\Support\Str::uuid(), // Random UUID that doesn't exist
            'event_id' => $this->event->id,
            'checksum' => 'fake-checksum',
        ]);

        $result = $this->ticketService->validateTicket($qrData);

        $this->assertFalse($result['valid']);
        $this->assertEquals('Ticket not found', $result['message']);
    }

    /** @test */
    public function it_rejects_event_mismatch()
    {
        $differentEvent = Event::factory()->create();

        $qrData = json_encode([
            'ticket_id' => $this->ticket->uuid,
            'event_id' => $differentEvent->id, // Wrong event
            'checksum' => $this->ticket->checksum,
        ]);

        $result = $this->ticketService->validateTicket($qrData);

        $this->assertFalse($result['valid']);
        $this->assertEquals('Event mismatch', $result['message']);
    }

    /** @test */
    public function it_rejects_already_checked_in_ticket()
    {
        // Mark ticket as already checked in
        $this->ticket->update(['is_checked_in' => true]);

        $qrData = json_encode([
            'ticket_id' => $this->ticket->uuid,
            'event_id' => $this->event->id,
            'checksum' => $this->ticket->checksum,
        ]);

        $result = $this->ticketService->validateTicket($qrData);

        $this->assertFalse($result['valid']);
        $this->assertEquals('Ticket already checked in', $result['message']);
    }

    /** @test */
    public function it_rejects_unpaid_order()
    {
        // Change order status to pending
        $this->order->update(['payment_status' => 'pending']);

        $qrData = json_encode([
            'ticket_id' => $this->ticket->uuid,
            'event_id' => $this->event->id,
            'checksum' => $this->ticket->checksum,
        ]);

        $result = $this->ticketService->validateTicket($qrData);

        $this->assertFalse($result['valid']);
        $this->assertEquals('Payment not completed', $result['message']);
    }

    /** @test */
    public function it_rejects_invalid_checksum()
    {
        $qrData = json_encode([
            'ticket_id' => $this->ticket->uuid,
            'event_id' => $this->event->id,
            'checksum' => 'fake-invalid-checksum',
        ]);

        $result = $this->ticketService->validateTicket($qrData);

        $this->assertFalse($result['valid']);
        $this->assertEquals('Invalid checksum', $result['message']);
    }

    /** @test */
    public function it_generates_unique_checksums_for_different_tickets()
    {
        // Create another ticket
        $order2 = Order::create([
            'order_number' => 'ORD-TEST456',
            'user_id' => $this->user->id,
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'amount' => 1000,
            'quantity' => 1,
            'total_amount' => 1000,
            'payment_status' => 'paid',
            'phone_number' => '0712345678',
        ]);

        $ticket2 = Ticket::factory()->create([
            'order_id' => $order2->id,
            'ticket_category_id' => $this->category->id,
        ]);

        $checksumReflection = new \ReflectionMethod($this->ticketService, 'generateChecksum');
        $checksumReflection->setAccessible(true);
        $checksum2 = $checksumReflection->invoke($this->ticketService, $ticket2);

        // Checksums should be different
        $this->assertNotEquals($this->ticket->checksum, $checksum2);
    }

    /** @test */
    public function checksum_is_deterministic()
    {
        // Generate checksum twice for same ticket
        $checksumReflection = new \ReflectionMethod($this->ticketService, 'generateChecksum');
        $checksumReflection->setAccessible(true);

        $checksum1 = $checksumReflection->invoke($this->ticketService, $this->ticket);
        $checksum2 = $checksumReflection->invoke($this->ticketService, $this->ticket);

        // Should be identical
        $this->assertEquals($checksum1, $checksum2);
    }

    /** @test */
    public function it_generates_svg_qr_codes()
    {
        // Generate tickets for a paid order
        $tickets = $this->ticketService->generateTickets($this->order);

        // Verify tickets were created
        $this->assertCount(1, $tickets);

        // Note: We can't easily test SVG output without actually generating it
        // But we can verify the method runs without error
        $this->assertTrue(true);
    }

    /** @test */
    public function generated_tickets_have_valid_structure()
    {
        $tickets = $this->ticketService->generateTickets($this->order);
        $ticket = $tickets->first();

        // Verify ticket has all required fields
        $this->assertNotNull($ticket->uuid);
        $this->assertNotNull($ticket->qr_code);
        $this->assertNotNull($ticket->checksum);
        $this->assertFalse($ticket->is_checked_in);
        $this->assertEquals($this->order->id, $ticket->order_id);

        // Verify QR code format
        $this->assertStringStartsWith('TKT-', $ticket->qr_code);
    }

    /** @test */
    public function each_ticket_gets_unique_qr_code()
    {
        // Create order with 5 tickets
        $order = Order::create([
            'order_number' => 'ORD-MULTI',
            'user_id' => $this->user->id,
            'event_id' => $this->event->id,
            'ticket_category_id' => $this->category->id,
            'amount' => 1000,
            'quantity' => 5,
            'total_amount' => 5000,
            'payment_status' => 'paid',
            'phone_number' => '0712345678',
        ]);

        $tickets = $this->ticketService->generateTickets($order);

        $qrCodes = $tickets->pluck('qr_code')->toArray();

        // All QR codes should be unique
        $this->assertCount(5, $qrCodes);
        $this->assertCount(5, array_unique($qrCodes));
    }

    /** @test */
    public function validation_works_with_failed_payment_status()
    {
        $this->order->update(['payment_status' => 'failed']);

        $qrData = json_encode([
            'ticket_id' => $this->ticket->uuid,
            'event_id' => $this->event->id,
            'checksum' => $this->ticket->checksum,
        ]);

        $result = $this->ticketService->validateTicket($qrData);

        $this->assertFalse($result['valid']);
        $this->assertEquals('Payment not completed', $result['message']);
    }

    /** @test */
    public function validation_accepts_paid_status()
    {
        // Ensure order is paid
        $this->order->update(['payment_status' => 'paid']);

        $qrData = json_encode([
            'ticket_id' => $this->ticket->uuid,
            'event_id' => $this->event->id,
            'checksum' => $this->ticket->checksum,
        ]);

        $result = $this->ticketService->validateTicket($qrData);

        $this->assertTrue($result['valid']);
    }
}
