<?php

namespace Tests\Feature;

use App\Models\Event;
use App\Models\TicketCategory;
use App\Models\TicketHold;
use App\Models\User;
use App\Services\TicketHoldService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Str;
use Tests\TestCase;

class TicketHoldSystemTest extends TestCase
{
    use RefreshDatabase;

    protected TicketHoldService $holdService;
    protected Event $event;
    protected TicketCategory $category;
    protected User $user;

    protected function setUp(): void
    {
        parent::setUp();

        $this->holdService = app(TicketHoldService::class);
        $this->user = User::factory()->create();

        $this->event = Event::factory()->create([
            'name' => 'Hold Test Event',
            'event_date' => now()->addDays(30),
        ]);

        $this->category = TicketCategory::create([
            'event_id' => $this->event->id,
            'name' => 'General',
            'price' => 1000,
            'quantity' => 50,
        ]);
    }

    /** @test */
    public function it_can_hold_tickets_for_a_session()
    {
        $sessionId = Str::uuid()->toString();
        $quantity = 5;

        $hold = $this->holdService->hold($this->category->id, $quantity, $sessionId);

        $this->assertNotNull($hold);
        $this->assertEquals($this->category->id, $hold->ticket_category_id);
        $this->assertEquals($quantity, $hold->quantity);
        $this->assertEquals($sessionId, $hold->session_id);
        $this->assertFalse($hold->is_released);
    }

    /** @test */
    public function it_updates_existing_hold_for_same_session()
    {
        $sessionId = Str::uuid()->toString();

        // Create initial hold of 3 tickets
        $firstHold = $this->holdService->hold($this->category->id, 3, $sessionId);

        // Update hold to 5 tickets
        $secondHold = $this->holdService->hold($this->category->id, 5, $sessionId);

        // Should be the same hold record, just updated
        $this->assertEquals($firstHold->id, $secondHold->id);
        $this->assertEquals(5, $secondHold->quantity);

        // Should only have one hold record
        $this->assertEquals(1, TicketHold::where('session_id', $sessionId)->count());
    }

    /** @test */
    public function it_extends_hold_expiration_time()
    {
        $sessionId = Str::uuid()->toString();

        $hold = $this->holdService->hold($this->category->id, 5, $sessionId);
        $originalExpiration = $hold->expires_at;

        // Wait a moment
        sleep(1);

        // Extend the hold
        $this->holdService->extendHold($sessionId);

        $hold->refresh();

        // New expiration should be later than original
        $this->assertTrue($hold->expires_at->isAfter($originalExpiration));
    }

    /** @test */
    public function it_calculates_available_quantity_correctly()
    {
        // Category has 50 tickets total
        // Hold 10 tickets
        $sessionId = Str::uuid()->toString();
        $this->holdService->hold($this->category->id, 10, $sessionId);

        // Should have 40 available (50 - 10)
        $available = $this->holdService->getAvailableQuantity($this->category->id);
        $this->assertEquals(40, $available);
    }

    /** @test */
    public function it_excludes_expired_holds_from_available_quantity()
    {
        // Category has 50 tickets
        $sessionId = Str::uuid()->toString();

        // Create a hold that's already expired
        $hold = $this->holdService->hold($this->category->id, 10, $sessionId);
        $hold->update(['expires_at' => now()->subMinutes(5)]);

        // Should have all 50 available since hold is expired
        $available = $this->holdService->getAvailableQuantity($this->category->id);
        $this->assertEquals(50, $available);
    }

    /** @test */
    public function it_checks_if_quantity_is_available()
    {
        // Category has 50 tickets
        // Hold 45 tickets
        $sessionId = Str::uuid()->toString();
        $this->holdService->hold($this->category->id, 45, $sessionId);

        // 5 available
        $this->assertTrue($this->holdService->isQuantityAvailable($this->category->id, 5));
        $this->assertFalse($this->holdService->isQuantityAvailable($this->category->id, 6));
    }

    /** @test */
    public function it_can_release_holds()
    {
        $sessionId = Str::uuid()->toString();

        $hold = $this->holdService->hold($this->category->id, 10, $sessionId);
        $this->assertFalse($hold->is_released);

        // Release the hold
        $this->holdService->release($sessionId);

        $hold->refresh();
        $this->assertTrue($hold->is_released);
    }

    /** @test */
    public function it_cleans_expired_holds()
    {
        // Create 3 holds: 1 active, 2 expired
        $activeSession = Str::uuid()->toString();
        $expiredSession1 = Str::uuid()->toString();
        $expiredSession2 = Str::uuid()->toString();

        // Active hold
        $activeHold = $this->holdService->hold($this->category->id, 5, $activeSession);

        // Expired holds
        $expiredHold1 = $this->holdService->hold($this->category->id, 5, $expiredSession1);
        $expiredHold1->update(['expires_at' => now()->subMinutes(5)]);

        $expiredHold2 = $this->holdService->hold($this->category->id, 5, $expiredSession2);
        $expiredHold2->update(['expires_at' => now()->subMinutes(10)]);

        // Clean expired holds
        $cleaned = $this->holdService->cleanExpiredHolds();

        // Should have cleaned 2 holds
        $this->assertEquals(2, $cleaned);

        // Verify they're marked as released
        $expiredHold1->refresh();
        $expiredHold2->refresh();
        $activeHold->refresh();

        $this->assertTrue($expiredHold1->is_released);
        $this->assertTrue($expiredHold2->is_released);
        $this->assertFalse($activeHold->is_released);
    }

    /** @test */
    public function it_returns_hold_expiration_time()
    {
        $sessionId = Str::uuid()->toString();

        $this->holdService->hold($this->category->id, 5, $sessionId);

        $expiration = $this->holdService->getHoldExpiration($sessionId);

        $this->assertNotNull($expiration);
        $this->assertTrue($expiration->isFuture());
    }

    /** @test */
    public function order_creation_creates_hold()
    {
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 5,
            ]);

        $response->assertStatus(201);
        $response->assertJsonStructure([
            'session_id',
            'hold_expires_at',
        ]);

        // Verify hold was created
        $sessionId = $response->json('session_id');
        $this->assertNotNull($sessionId);

        $hold = TicketHold::where('session_id', $sessionId)->first();
        $this->assertNotNull($hold);
        $this->assertEquals(5, $hold->quantity);
    }

    /** @test */
    public function holds_prevent_overselling()
    {
        // Category has 50 tickets
        // User 1 holds 48 tickets
        $session1 = Str::uuid()->toString();
        $this->holdService->hold($this->category->id, 48, $session1);

        // User 2 tries to purchase 5 tickets (should fail - only 2 available)
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 5,
            ]);

        $response->assertStatus(422);
        $response->assertJsonFragment([
            'available_quantity' => 2,
        ]);
    }

    /** @test */
    public function expired_holds_do_not_block_purchases()
    {
        // Create an expired hold of 48 tickets
        $expiredSession = Str::uuid()->toString();
        $hold = $this->holdService->hold($this->category->id, 48, $expiredSession);
        $hold->update(['expires_at' => now()->subMinutes(5)]);

        // Should be able to purchase since hold is expired
        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/orders', [
                'event_id' => $this->event->id,
                'ticket_category_id' => $this->category->id,
                'phone_number' => '0712345678',
                'quantity' => 10,
            ]);

        $response->assertStatus(201);
    }

    /** @test */
    public function it_handles_multiple_concurrent_holds()
    {
        // Category has 50 tickets
        // Create 5 different sessions each holding 5 tickets
        for ($i = 0; $i < 5; $i++) {
            $sessionId = Str::uuid()->toString();
            $this->holdService->hold($this->category->id, 5, $sessionId);
        }

        // Total holds: 25 tickets
        // Available: 25 tickets
        $available = $this->holdService->getAvailableQuantity($this->category->id);
        $this->assertEquals(25, $available);

        // Should be able to purchase 25 more
        $this->assertTrue($this->holdService->isQuantityAvailable($this->category->id, 25));

        // Should NOT be able to purchase 26
        $this->assertFalse($this->holdService->isQuantityAvailable($this->category->id, 26));
    }

    /** @test */
    public function hold_expiration_defaults_to_10_minutes()
    {
        $sessionId = Str::uuid()->toString();

        $hold = $this->holdService->hold($this->category->id, 5, $sessionId);

        $expectedExpiration = now()->addMinutes(10);

        // Allow 5 second tolerance for test execution time
        $this->assertTrue(
            $hold->expires_at->between(
                $expectedExpiration->copy()->subSeconds(5),
                $expectedExpiration->copy()->addSeconds(5)
            )
        );
    }
}
